package ru.yandex.solomon.roles;

import java.time.Clock;
import java.util.List;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.idm.IdmClient;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.acl.db.AclDaoContext;
import ru.yandex.solomon.acl.db.GroupMemberDao;
import ru.yandex.solomon.acl.db.ProjectAclEntryDao;
import ru.yandex.solomon.acl.db.ServiceProviderAclEntryDao;
import ru.yandex.solomon.acl.db.SystemAclEntryDao;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.protobuf.StaffClientConfig;
import ru.yandex.solomon.config.protobuf.TIdmConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.locks.LockService;
import ru.yandex.solomon.roles.idm.IdmContext;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.staff.StaffClient;

/**
 * @author Alexey Trushkin
 */
@Import({
        IdmContext.class,
        AclDaoContext.class,
        LockContext.class,
})
@ConditionalOnBean(StaffClientConfig.class)
@Configuration
public class RolesContext {

    @Bean
    public RoleManager projectRoleManager(
            ProjectAclEntryDao projectAclEntryDao,
            SolomonConfHolder holder,
            ProjectsDao projectsDao)
    {
        return new ProjectRoleManager(projectAclEntryDao, holder, projectsDao);
    }

    @Bean
    public RoleManager serviceProviderRoleManager(
            ServiceProviderAclEntryDao serviceProviderAclEntryDao,
            ServiceProvidersDao serviceProvidersDao)
    {
        return new ServiceProviderRoleManager(serviceProviderAclEntryDao, serviceProvidersDao);
    }

    @Bean
    public RoleManager systemRoleManager(SystemAclEntryDao systemAclEntryDao) {
        return new SystemRoleManager(systemAclEntryDao);
    }

    @Bean
    public GroupMemberManager groupMemberManager(Optional<StaffClient> staffClient, GroupMemberDao groupMemberDao) {
        if (staffClient.isEmpty()) {
            return null;
        }
        return new GroupMemberManager(groupMemberDao, staffClient.get());
    }

    @Bean
    public RoleAclSynchronizer roleAclSynchronizer(
            ProjectsDao projectsDao,
            Optional<TIdmConfig> config,
            Optional<IdmClient> idmClient
    ) {
        if (config.isEmpty() || idmClient.isEmpty()) {
            return null;
        }
        return new RoleAclSynchronizer(projectsDao, config.get(), idmClient.get());
    }

    @Bean
    public GroupMembersUpdateScheduler groupMembersUpdateScheduler(
            StaffClient staffClient,
            GroupMemberDao groupMemberDao,
            Optional<TIdmConfig> config,
            @Qualifier("gatewayLockService") LockService lockService,
            ThreadPoolProvider threads,
            MetricRegistry registry)
    {
        if (config.isEmpty()) {
            return null;
        }
        var lock = lockService.distributedLock("GroupMemberUpdaterMaster");
        var timer = threads.getSchedulerExecutorService();
        var interval = TimeUnitConverter.millis(config.get().getGroupMembersRefreshInterval());
        return new GroupMembersUpdateScheduler(groupMemberDao, staffClient, lock, threads.getIOExecutor(), timer, Clock.systemUTC(), registry, interval);
    }

    @Bean
    public RoleAclSynchronizerListener roleAclSynchronizerListener(
            Optional<RoleAclSynchronizer> roleAclSynchronizer,
            @Qualifier("gatewayLockService") LockService lockService,
            ThreadPoolProvider threads)
    {
        if (roleAclSynchronizer.isEmpty()) {
            return null;
        }
        var lock = lockService.distributedLock("RoleAclSynchronizerListener");
        return new RoleAclSynchronizerListener(threads.getIOExecutor(), lock, roleAclSynchronizer.get());
    }

    @Bean
    public RoleService roleService(List<RoleManager> managers, List<RoleChangeListener> listeners) {
        return new RoleServiceImpl(managers, listeners);
    }
}
