package ru.yandex.solomon.roles;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import ru.yandex.solomon.acl.db.ServiceProviderAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ServiceProviderAclEntry;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.roles.idm.IdmException;
import ru.yandex.solomon.roles.idm.dto.IdmAddRoleDto;
import ru.yandex.solomon.roles.idm.dto.IdmRemoveRoleDto;
import ru.yandex.solomon.roles.idm.dto.IdmResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRoleTreeResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRolesPageResponseDto;
import ru.yandex.solomon.roles.idm.dto.RoleDto;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Alexey Trushkin
 */
public class ServiceProviderRoleManager implements RoleManager {
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(3)
            .withDelay(1_000)
            .withMaxDelay(60_000);
    private final ServiceProviderAclEntryDao dao;
    private final ServiceProvidersDao serviceProvidersDao;

    public ServiceProviderRoleManager(
            ServiceProviderAclEntryDao serviceProviderAclEntryDao,
            ServiceProvidersDao serviceProvidersDao)
    {
        this.dao = serviceProviderAclEntryDao;
        this.serviceProvidersDao = serviceProvidersDao;
    }


    @Override
    public boolean accepts(RoleDto role) {
        return role.isServiceProvider();
    }

    @Override
    public CompletableFuture<IdmResponseDto.ResultData> addRole(IdmAddRoleDto idmAddRoleDto) {
        var serviceProviderId = idmAddRoleDto.serviceProviderId;
        return serviceProvidersDao.read(serviceProviderId).thenCompose(serviceProvider -> {
            if (serviceProvider.isEmpty()) {
                return CompletableFuture.failedFuture(new IdmException("Hasn't service provider " + serviceProviderId, true));
            }
            return addRole(idmAddRoleDto, serviceProviderId);
        });
    }

    private CompletableFuture<IdmResponseDto.ResultData> addRole(IdmAddRoleDto idmAddRoleDto, String serviceProviderId) {
        var uid = idmAddRoleDto.getUid();
        var role = idmAddRoleDto.role.roleId;
        return RetryCompletableFuture.runWithRetries(() -> addRole(idmAddRoleDto, serviceProviderId, uid, role), RETRY_CONFIG);
    }

    private CompletableFuture<IdmResponseDto.ResultData> addRole(
            IdmAddRoleDto idmAddRoleDto,
            String serviceProviderId,
            String uid,
            String role)
    {
        return dao.find(serviceProviderId, uid, idmAddRoleDto.aclUidType)
                .thenCompose(entryOptional -> {
                    var entry = prepareEntry(entryOptional, serviceProviderId, idmAddRoleDto.aclUidType, uid, role);
                    if (entryOptional.isPresent()) {
                        return dao.update(entry);
                    } else {
                        return createNewEntry(entry);
                    }
                })
                .thenApply(unused -> IdmResponseDto.ResultData.ofServiceProvider(serviceProviderId));
    }

    @Override
    public CompletableFuture<Void> removeRole(IdmRemoveRoleDto idmRemoveRoleDto) {
        var serviceProviderId = idmRemoveRoleDto.serviceProviderId;
        var uid = idmRemoveRoleDto.getUid();
        var role = idmRemoveRoleDto.role.roleId;
        return RetryCompletableFuture.runWithRetries(() -> removeRole(idmRemoveRoleDto, serviceProviderId, uid, role), RETRY_CONFIG);
    }

    private CompletableFuture<Void> removeRole(
            IdmRemoveRoleDto idmRemoveRoleDto,
            String serviceProviderId,
            String uid,
            String role)
    {
        return dao.find(serviceProviderId, uid, idmRemoveRoleDto.aclUidType)
                .thenCompose(entryOptional -> {
                    if (entryOptional.isPresent()) {
                        var entry = entryOptional.get();
                        entry.getRoles().remove(role);
                        if (entry.getRoles().isEmpty()) {
                            return deleteEntry(entry);
                        } else {
                            return dao.update(entry);
                        }
                    } else {
                        return CompletableFuture.completedFuture(null);
                    }
                });
    }

    @Override
    public CompletableFuture<IdmRoleTreeResponseDto.RoleSubTree> getRoleSubTree() {
        var tree = IdmRoleTreeResponseDto.RoleSubTree.of(
                RoleDto.Type.SERVICE_PROVIDER, "Роль сервис-провайдера", "Service provider role");
        var field = IdmRoleTreeResponseDto.Field.of("serviceProvider",
                "Идентификатор сервис-провайдера", "Service provider identifier", "charfield", true);
        tree.fields = List.of(field);
        tree.addRole(Role.SERVICE_PROVIDER_ADMIN, "Администратор сервис-провайдера", "Service provider admin");
        tree.addRole(Role.SERVICE_PROVIDER_ALERT_EDITOR, "Редактор алертов сервис-провайдера", "Service provider alerts editor");
        return CompletableFuture.completedFuture(tree);
    }

    @Override
    public CompletableFuture<List<IdmRolesPageResponseDto.Role>> getRoles() {
        return dao.getAll()
                .thenApply(entries -> {
                    List<IdmRolesPageResponseDto.Role> roles = new ArrayList<>(entries.size());
                    for (var entry : entries) {
                        roles.addAll(IdmRolesPageResponseDto.Role.createRoles(entry));
                    }
                    return roles;
                });
    }

    private CompletableFuture<Void> deleteEntry(ServiceProviderAclEntry entry) {
        return dao.delete(entry).thenAccept(result -> validateDaoResult(result, entry));
    }

    private CompletableFuture<Void> createNewEntry(ServiceProviderAclEntry entry) {
        return dao.create(entry).thenAccept(result -> validateDaoResult(result, entry));
    }

    private ServiceProviderAclEntry prepareEntry(
            Optional<ServiceProviderAclEntry> optional,
            String serviceProviderId,
            AclUidType type,
            String uid,
            String role)
    {
        if (optional.isPresent()) {
            ServiceProviderAclEntry entry = optional.get();
            entry.getRoles().add(role);
            return entry;
        }
        return new ServiceProviderAclEntry(serviceProviderId, uid, type, Set.of(role), 0);
    }

    private void validateDaoResult(Boolean result, ServiceProviderAclEntry entry) {
        if (!result) {
            String message = String.format(
                    "ServiceProviderAclEntry (%s) with version %s is out of date",
                    entry.getCompositeId(),
                    entry.getVersion()
            );
            throw new ConflictException(message);
        }
    }

}
