package ru.yandex.solomon.roles;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import ru.yandex.solomon.acl.db.SystemAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.SystemAclEntry;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.roles.idm.dto.IdmAddRoleDto;
import ru.yandex.solomon.roles.idm.dto.IdmRemoveRoleDto;
import ru.yandex.solomon.roles.idm.dto.IdmResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRoleTreeResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRolesPageResponseDto;
import ru.yandex.solomon.roles.idm.dto.RoleDto;

/**
 * @author Alexey Trushkin
 */
public class SystemRoleManager implements RoleManager {

    private final SystemAclEntryDao dao;

    public SystemRoleManager(SystemAclEntryDao dao) {
        this.dao = dao;
    }

    @Override
    public boolean accepts(RoleDto role) {
        return role.isSystem();
    }

    @Override
    public CompletableFuture<IdmResponseDto.ResultData> addRole(IdmAddRoleDto idmAddRoleDto) {
        var uid = idmAddRoleDto.getUid();
        var role = idmAddRoleDto.role.roleId;
        return dao.find(uid, idmAddRoleDto.aclUidType)
                .thenCompose(entryOptional -> {
                    var entry = prepareEntry(entryOptional, idmAddRoleDto.aclUidType, uid, role);
                    if (entryOptional.isPresent()) {
                        return dao.update(entry);
                    } else {
                        return createNewEntry(entry);
                    }
                })
                .thenApply(unused -> null);
    }

    @Override
    public CompletableFuture<Void> removeRole(IdmRemoveRoleDto idmRemoveRoleDto) {
        var uid = idmRemoveRoleDto.getUid();
        var role = idmRemoveRoleDto.role.roleId;
        return dao.find(uid, idmRemoveRoleDto.aclUidType)
                .thenCompose(entryOptional -> {
                    if (entryOptional.isPresent()) {
                        var entry = entryOptional.get();
                        entry.getRoles().remove(role);
                        if (entry.getRoles().isEmpty()) {
                            return deleteEntry(entry);
                        } else {
                            return dao.update(entry);
                        }
                    } else {
                        return CompletableFuture.completedFuture(null);
                    }
                });
    }

    @Override
    public CompletableFuture<IdmRoleTreeResponseDto.RoleSubTree> getRoleSubTree() {
        var tree = IdmRoleTreeResponseDto.RoleSubTree.of(
                RoleDto.Type.SYSTEM, "Системная роль", "System role");
        tree.addRole(Role.ADMIN, "Администратор мониторинга", "Admin");
        tree.addRole(Role.BETA_TESTER, "Бета-тестирование UI", "Beta-tester", false);
        tree.addRole(Role.PROJECT_DELETER, "Удаление проектов", "Project deleter", false);
        return CompletableFuture.completedFuture(tree);
    }

    @Override
    public CompletableFuture<List<IdmRolesPageResponseDto.Role>> getRoles() {
        return dao.getAll()
                .thenApply(entries -> {
                    List<IdmRolesPageResponseDto.Role> roles = new ArrayList<>(entries.size());
                    for (var entry : entries) {
                        roles.addAll(IdmRolesPageResponseDto.Role.createRoles(entry));
                    }
                    return roles;
                });
    }

    private CompletableFuture<Void> deleteEntry(SystemAclEntry entry) {
        return dao.delete(entry).thenAccept(result -> validateDaoResult(result, entry));
    }

    private CompletableFuture<Void> createNewEntry(SystemAclEntry entry) {
        return dao.create(entry).thenAccept(result -> validateDaoResult(result, entry));
    }

    private SystemAclEntry prepareEntry(Optional<SystemAclEntry> optional, AclUidType type, String uid, String role) {
        if (optional.isPresent()) {
            SystemAclEntry entry = optional.get();
            entry.getRoles().add(role);
            return entry;
        }
        return new SystemAclEntry(uid, type, Set.of(role), 0);
    }

    private void validateDaoResult(Boolean result, SystemAclEntry entry) {
        if (!result) {
            String message = String.format(
                    "SystemAclEntry (%s) with version %s is out of date",
                    entry.getCompositeId(),
                    entry.getVersion()
            );
            throw new ConflictException(message);
        }
    }

}
