package ru.yandex.solomon.roles.idm;

import java.util.Optional;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.idm.IdmClient;
import ru.yandex.idm.IdmClientOptions;
import ru.yandex.idm.http.HttpIdmClient;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.TIdmConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.util.SolomonEnv;

import static ru.yandex.solomon.config.OptionalSet.setString;

/**
 * @author Alexey Trushkin
 */
@Import({
        IdmController.class,
})
@Configuration
public class IdmContext {
    private static final Logger logger = LoggerFactory.getLogger(IdmContext.class);

    @Bean
    public IdmDtoConverter idmDtoConverter() {
        return new IdmDtoConverter();
    }

    @Bean
    public IdmMigrationService idmMigrationService(Optional<IdmClient> idmClient, SolomonConfHolder holder, Optional<TIdmConfig> config) {
        if (config.isEmpty() || idmClient.isEmpty()) {
            return null;
        }
        return new IdmMigrationService(idmClient.get(), holder, config.get());
    }

    @Bean
    @Nullable
    public IdmClient idmClient(
            ThreadPoolProvider threadPoolProvider,
            MetricRegistry registry,
            Optional<TIdmConfig> config,
            SecretProvider secretProvider)
    {
        if (config.isEmpty()) {
            return null;
        }
        Optional<String> token = secretProvider.getSecret(config.get().getOAuthToken());
        if (SolomonEnv.DEVELOPMENT.isActive() && token.isEmpty()) {
            logger.warn("cannot create IdmClient without oauth token");
            return null;
        }

        IdmClientOptions.Builder opts = IdmClientOptions.newBuilder().setMetricRegistry(registry);
        setString(opts::setUrl, config.get().getUrl());
        opts.setOauthToken(token.orElseThrow(() -> new RuntimeException("cannot create IdmClient without oauth token")));
        opts.setExecutor(threadPoolProvider.getExecutorService("CpuLowPriority", ""));
        return new HttpIdmClient(opts.build());
    }

    @Bean
    @Nullable
    public IdmProjectListener idmProjectListener(
            Optional<IdmMigrationService> idmMigrationService,
            ThreadPoolProvider threadPoolProvider)
    {
        if (idmMigrationService.isEmpty()) {
            return null;
        }
        return new IdmProjectListener(
                idmMigrationService.get(),
                threadPoolProvider.getExecutorService("CpuLowPriority", ""));
    }
}
