package ru.yandex.solomon.roles.idm;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.http.MediaType;
import org.springframework.util.MultiValueMap;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.server.ServerWebExchange;

import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.SolomonTeam;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.config.protobuf.TIdmConfig;
import ru.yandex.solomon.exception.handlers.CommonApiExceptionHandler;
import ru.yandex.solomon.roles.RoleService;
import ru.yandex.solomon.roles.idm.dto.IdmMigrateProjectsDto;
import ru.yandex.solomon.roles.idm.dto.IdmResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRoleTreeResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRolesPageResponseDto;
import ru.yandex.solomon.spring.ConditionalOnBean;

import static ru.yandex.solomon.roles.idm.dto.IdmResponseDto.NO_AUTH_CODE;

/**
 * @author Alexey Trushkin
 */
@RestController
@RequestMapping(path = "/api/idm")
@ConditionalOnBean(TIdmConfig.class)
@ParametersAreNonnullByDefault
public class IdmController {
    private static final Logger logger = LoggerFactory.getLogger(IdmController.class);

    private final RoleService roleService;
    private final IdmDtoConverter converter;
    private final TIdmConfig config;
    private final IdmMigrationService idmMigrationService;
    private final MetricRegistry metricRegistry;
    private final ConcurrentMap<String, Rate> metricsMap = new ConcurrentHashMap<>();

    public IdmController(
            RoleService roleService,
            IdmDtoConverter converter,
            TIdmConfig config,
            Optional<IdmMigrationService> idmMigrationService,
            MetricRegistry metricRegistry)
    {
        this.roleService = roleService;
        this.converter = converter;
        this.config = config;
        this.idmMigrationService = idmMigrationService.orElse(null);
        this.metricRegistry = metricRegistry;
    }

    @PostMapping(
            value = "/add-role",
            consumes = MediaType.APPLICATION_FORM_URLENCODED_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public CompletableFuture<IdmResponseDto> addRole(
            @RequireAuth AuthSubject subject,
            @RequestHeader(value = "X-IDM-Request-Id", required = false) String idmRequestId,
            ServerWebExchange exchange)
    {
        if (!validateAuth(subject)) {
            return CompletableFuture.completedFuture(IdmResponseDto.error(NO_AUTH_CODE, subject.getUniqueId() + " not authorized"));
        }

        return exchange.getFormData()
                .toFuture()
                .thenApply(data -> logRequest(data, idmRequestId))
                .thenApply(converter::newIdmAddRoleDto)
                .thenCompose(roleService::addRole)
                .thenApply(this::returnOk)
                .exceptionally(throwable -> returnError(throwable, idmRequestId, "/api/idm/add-role", "POST"));
    }

    @PostMapping(
            value = "/remove-role",
            consumes = MediaType.APPLICATION_FORM_URLENCODED_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public CompletableFuture<IdmResponseDto> removeRole(
            @RequireAuth AuthSubject subject,
            @RequestHeader(value = "X-IDM-Request-Id", required = false) String idmRequestId,
            ServerWebExchange exchange)
    {
        if (!validateAuth(subject)) {
            return CompletableFuture.completedFuture(IdmResponseDto.error(NO_AUTH_CODE, subject.getUniqueId() + " not authorized"));
        }

        return exchange.getFormData()
                .toFuture()
                .thenApply(data -> logRequest(data, idmRequestId))
                .thenApply(converter::newIdmRemoveRoleDto)
                .thenCompose(roleService::removeRole)
                .thenApply(this::returnOk)
                .exceptionally(throwable -> returnError(throwable, idmRequestId, "/api/idm/remove-role", "POST"));
    }

    @GetMapping(value = "/info", produces = MediaType.APPLICATION_JSON_VALUE)
    public CompletableFuture<IdmResponseDto> roleTree(
            @RequireAuth AuthSubject subject,
            @RequestHeader(value = "X-IDM-Request-Id", required = false) String idmRequestId)
    {
        if (!validateAuth(subject)) {
            return CompletableFuture.completedFuture(IdmResponseDto.error(NO_AUTH_CODE, subject.getUniqueId() + " not authorized"));
        }

        return roleService.getRoleTree()
                .thenApply(roleTree -> (IdmResponseDto) IdmRoleTreeResponseDto.ok(roleTree))
                .exceptionally(throwable -> returnError(throwable, idmRequestId, "/api/idm/info", "GET"));
    }

    @GetMapping(value = "/get-roles", produces = MediaType.APPLICATION_JSON_VALUE)
    public CompletableFuture<IdmResponseDto> getRoles(
            @RequireAuth AuthSubject subject,
            @RequestHeader(value = "X-IDM-Request-Id", required = false) String idmRequestId)
    {
        if (!validateAuth(subject)) {
            return CompletableFuture.completedFuture(IdmResponseDto.error(NO_AUTH_CODE, subject.getUniqueId() + " not authorized"));
        }

        return roleService.getRoles()
                .thenApply(roles -> (IdmResponseDto) IdmRolesPageResponseDto.ok(roles))
                .exceptionally(throwable -> returnError(throwable, idmRequestId, "/api/idm/get-roles", "GET"));
    }

    @PostMapping(
            value = "/migrate-project",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public CompletableFuture<Void> migrateProjects(
            @RequireAuth AuthSubject subject,
            @RequestBody IdmMigrateProjectsDto idmMigrateProjectsDto)
    {
        if (!SolomonTeam.isMember(subject)) {
            return CompletableFuture.failedFuture(new AuthorizationException("Not authorized for project migration"));
        }

        return idmMigrationService.migrate(idmMigrateProjectsDto);
    }

    private boolean validateAuth(AuthSubject subject) {
        if (AuthType.TvmService.equals(subject.getAuthType())) {
            return subject.getUniqueId().equals(config.getTvmClientId());
        }
        //just for tests
        return SolomonTeam.isMember(subject);
    }

    private IdmResponseDto returnOk(IdmResponseDto.ResultData data) {
        return IdmResponseDto.ok(data);
    }

    private IdmResponseDto returnOk(Void unused) {
        return IdmResponseDto.ok();
    }

    private IdmResponseDto returnError(Throwable throwable, @Nullable String idmRequestId, String endpoint, String method) {
        logger.error("Error while processing idm request(" + idmRequestId + ") ", throwable);
        var data = CommonApiExceptionHandler.tryHandleSolomonException(throwable, idmRequestId, false);
        incrementErrorMetric(data.getHttpStatus().value(), endpoint, method);
        if (throwable instanceof IdmException e) {
            return IdmResponseDto.error(data.getHttpStatus().value(), e);
        }
        return IdmResponseDto.error(data.getHttpStatus().value(), throwable.getMessage());
    }

    private MultiValueMap<String, String> logRequest(MultiValueMap<String, String> data, @Nullable String idmRequestId) {
        logger.info("Idm Request({}) data {}", idmRequestId, data);
        return data;
    }

    private void incrementErrorMetric(int status, String endpoint, String method) {
        Rate rate = metricsMap.computeIfAbsent(endpoint + status, s -> {
            Labels labels = Labels.of("endpoint", endpoint, "method", method, "code", Integer.toString(status));
            return metricRegistry.rate("idm.server.requests.errors", labels);
        });
        rate.inc();
    }
}
