package ru.yandex.solomon.roles.idm;

import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.stereotype.Component;
import org.springframework.util.MultiValueMap;

import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.roles.idm.dto.IdmAddRoleDto;
import ru.yandex.solomon.roles.idm.dto.IdmOperationDto;
import ru.yandex.solomon.roles.idm.dto.IdmRemoveRoleDto;
import ru.yandex.solomon.roles.idm.dto.RoleDto;

/**
 * @author Alexey Trushkin
 */
@Component
public class IdmDtoConverter {
    private static final String TRUE_VALUE = "1";
    private final ObjectMapper objectMapper;

    public IdmDtoConverter() {
        this.objectMapper = new ObjectMapper();
    }

    public IdmAddRoleDto newIdmAddRoleDto(MultiValueMap<String, String> data) {
        IdmAddRoleDto result = new IdmAddRoleDto();
        fillData(data, result);
        return result;
    }

    public IdmRemoveRoleDto newIdmRemoveRoleDto(MultiValueMap<String, String> data) {
        IdmRemoveRoleDto result = new IdmRemoveRoleDto();
        fillData(data, result);
        result.delete = TRUE_VALUE.equals(data.getFirst("fired")) ||
                TRUE_VALUE.equals(data.getFirst("deleted"));
        return result;
    }

    private void fillData(MultiValueMap<String, String> data, IdmOperationDto result) {
        result.login = data.getFirst("login");
        result.group = data.getFirst("group");
        result.subjectType = data.getFirst("subject_type");

        if (result.isUser()) {
            result.aclUidType = AclUidType.USER;
        } else if (result.isTvm()) {
            result.aclUidType = AclUidType.TVM;
        } else if (result.group != null) {
            result.aclUidType = AclUidType.GROUP;
        } else {
            throw new IdmException(String.format("Unknown subject_type: %s in %s", result.subjectType, data));
        }
        try {
            result.role = objectMapper.readValue(data.getFirst("role"), RoleDto.class);
            if (result.role.isProject()) {
                var type = new TypeReference<Map<String, Object>>() {
                };
                Map<String, Object> fields = objectMapper.readValue(data.getFirst("fields"), type);
                result.project = fields.get("project").toString();
            } else if (result.role.isServiceProvider()) {
                var type = new TypeReference<Map<String, Object>>() {
                };
                Map<String, Object> fields = objectMapper.readValue(data.getFirst("fields"), type);
                result.serviceProviderId = fields.get("serviceProvider").toString();
            }
        } catch (Throwable e) {
            throw new IdmException("Can't parse role: " + e.getMessage());
        }
    }
}
