package ru.yandex.solomon.roles.idm;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Throwables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.idm.IdmClient;
import ru.yandex.idm.dto.RoleRequestDto;
import ru.yandex.idm.http.HttpIdmClient;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.config.protobuf.TIdmConfig;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.roles.idm.dto.IdmMigrateProjectsDto;

/**
 * @author Alexey Trushkin
 */
public class IdmMigrationService {

    private static final Logger logger = LoggerFactory.getLogger(IdmMigrationService.class);

    private final IdmClient idmClient;
    private final SolomonConfHolder holder;
    private final TIdmConfig config;

    public IdmMigrationService(IdmClient idmClient, SolomonConfHolder holder, TIdmConfig config) {
        this.idmClient = idmClient;
        this.holder = holder;
        this.config = config;
    }

    public CompletableFuture<Void> migrate(IdmMigrateProjectsDto idmMigrateProjectsDto) {
        if (idmMigrateProjectsDto.projects.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        CompletableFuture<Void> result = new CompletableFuture<>();
        Iterator<String> iterator = idmMigrateProjectsDto.projects.iterator();
        requestNext(iterator, result);
        return result;
    }

    private void requestNext(Iterator<String> projects, CompletableFuture<Void> result) {
        try {
            String projectId = projects.next();
            SolomonConfWithContext conf = holder.getConfOrThrow();
            Project project = conf.getProject(projectId);
            if (project == null) {
                logger.error("Project not found " + projectId);
                onResponse(null, projects, result);
                return;
            }
            new ProjectRoleRequester(project).start()
                    .whenComplete((Void unused, Throwable e) -> onResponse(e, projects, result));
        } catch (Throwable throwable) {
            result.completeExceptionally(throwable);
        }
    }

    private void onResponse(Throwable e, Iterator<String> projects, CompletableFuture<Void> result) {
        if (e != null) {
            result.completeExceptionally(e);
            return;
        }

        if (projects.hasNext()) {
            requestNext(projects, result);
        } else {
            result.complete(null);
        }
    }

    public CompletableFuture<Void> migrateDeletedProjects(List<String> removedProjects) {
        if (removedProjects.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        return new ProjectRoleDeleter(removedProjects).start();
    }

    private class ProjectRoleRequester {

        private final Iterator<RoleRecord> iterator;
        private final Project project;
        private final CompletableFuture<Void> result = new CompletableFuture<>();

        public ProjectRoleRequester(Project project) {
            this.project = project;
            Map<String, Set<Role>> roles = new HashMap<>();
            addRole(roles, project.getOwner(), Role.PROJECT_ADMIN);
            Acl acl = project.getAcl();
            acl.getCanDelete().forEach(s -> addRole(roles, s, Role.PROJECT_ADMIN));
            acl.getCanUpdate().forEach(s -> addRole(roles, s, Role.PROJECT_ADMIN));
            acl.getCanRead().forEach(s -> addRole(roles, s, Role.VIEWER));
            acl.getCanWrite().forEach(s -> {
                addRole(roles, s, Role.VIEWER);
                addRole(roles, s, Role.PUSHER);
            });
            iterator = roles.entrySet().stream()
                    .flatMap(entry -> entry.getValue().stream().map(v -> new RoleRecord(entry.getKey(), v)))
                    .iterator();
        }

        private void addRole(Map<String, Set<Role>> roles, String user, Role role) {
            Set<Role> set = roles.getOrDefault(user, new HashSet<>());
            set.add(role);
            roles.put(user, set);
        }

        public CompletableFuture<Void> start() {
            if (!iterator.hasNext()) {
                result.complete(null);
                return result;
            }
            requestNext();
            return result;
        }

        private void requestNext() {
            RoleRecord next = iterator.next();
            idmClient.requestRole(createRequest(next.login(), next.role()))
                    .whenComplete(this::onResponse);
        }

        private void onResponse(Void unused, Throwable e) {
            // skip fired and failed by client
            if (e != null
                    && !(Throwables.getRootCause(e) instanceof HttpIdmClient.FiredError)
                    && !(Throwables.getRootCause(e) instanceof HttpIdmClient.ClientError))
            {
                result.completeExceptionally(e);
                return;
            }

            if (iterator.hasNext()) {
                requestNext();
            } else {
                result.complete(null);
            }
        }

        private RoleRequestDto createRequest(String user, Role role) {
            boolean isTvm = user.startsWith("tvm-");
            if (isTvm) {
                user = user.substring(4);
            }
            return RoleRequestDto.newProjectRoleRequest(
                    user,
                    project.getId(),
                    role.name(),
                    config.getSystemName(),
                    isTvm);
        }

    }

    private class ProjectRoleDeleter {

        private final Iterator<String> iterator;
        private final CompletableFuture<Void> result = new CompletableFuture<>();

        public ProjectRoleDeleter(List<String> removedProjects) {
            iterator = removedProjects.iterator();
        }

        public CompletableFuture<Void> start() {
            if (!iterator.hasNext()) {
                result.complete(null);
                return result;
            }
            requestNext();
            return result;
        }

        private void requestNext() {
            String next = iterator.next();
            idmClient.deleteRoles(createRequest(next))
                    .whenComplete(this::onResponse);
        }

        private void onResponse(Void unused, Throwable e) {
            if (e != null) {
                result.completeExceptionally(e);
                return;
            }

            if (iterator.hasNext()) {
                requestNext();
            } else {
                result.complete(null);
            }
        }

        private RoleRequestDto createRequest(String projectId) {
            return RoleRequestDto.newProjectRoleRequest(
                    null,
                    projectId,
                    null,
                    config.getSystemName(),
                    false);
        }
    }

    private record RoleRecord(String login, Role role) {
    }
}
