package ru.yandex.solomon.roles.idm;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.watch.SolomonConfListener;
import ru.yandex.solomon.roles.idm.dto.IdmMigrateProjectsDto;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class IdmProjectListener implements SolomonConfListener {

    private static final Logger logger = LoggerFactory.getLogger(IdmProjectListener.class);
    private final IdmMigrationService idmMigrationService;
    private final Set<String> knownProjects;

    private final ActorWithFutureRunner actor;
    private volatile SolomonConfWithContext conf;

    public IdmProjectListener(
            IdmMigrationService idmMigrationService,
            ExecutorService executor)
    {
        this.idmMigrationService = idmMigrationService;
        knownProjects = new HashSet<>();
        this.actor = new ActorWithFutureRunner(this::tryMigrate, executor);
    }

    @Override
    public void onConfigurationLoad(SolomonConfWithContext conf) {
        this.conf = conf;
        actor.schedule();
    }

    private CompletableFuture<?> tryMigrate() {
        try {
            IdmMigrateProjectsDto dto = new IdmMigrateProjectsDto();
            dto.projects = conf.projects().stream()
                    .filter(projectId -> !knownProjects.contains(projectId))
                    .collect(Collectors.toSet());
            List<String> removedProjects = new ArrayList<>();
            for (String projectId : knownProjects) {
                if (conf.getProject(projectId) == null) {
                    removedProjects.add(projectId);
                }
            }
            if (dto.projects.isEmpty() && removedProjects.isEmpty()) {
                return completedFuture(null);
            }
            if (knownProjects.isEmpty()) {
                // initial state
                knownProjects.addAll(dto.projects);
                return CompletableFuture.completedFuture(null);
            }

            return idmMigrationService.migrate(dto)
                    .thenCompose(unused -> idmMigrationService.migrateDeletedProjects(removedProjects))
                    .handle((unused, throwable) -> {
                        if (throwable != null) {
                            logger.error("Error while migrate projects.", throwable);
                        } else {
                            knownProjects.addAll(dto.projects);
                            removedProjects.forEach(knownProjects::remove);
                        }
                        return unused;
                    });
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }
}
