package ru.yandex.solomon.roles.idm.dto;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.solomon.auth.roles.Permission;

/**
 * @author Alexey Trushkin
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class IdmRoleTreeResponseDto extends IdmResponseDto {

    @JsonProperty("roles")
    public RoleTree roles;

    public static IdmRoleTreeResponseDto ok(RoleTree roles) {
        var result = new IdmRoleTreeResponseDto();
        result.roles = roles;
        result.code = 0;
        return result;
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class RoleTree {

        @JsonProperty("slug")
        public String slug;

        @JsonProperty("name")
        public StringMultiLanguage name;

        @JsonProperty("values")
        public Map<String, RoleSubTree> values = new HashMap<>();

        public static RoleTree of(String slug, String nameRu, String nameEn) {
            RoleTree tree = new RoleTree();
            tree.name = StringMultiLanguage.of(nameRu, nameEn);
            tree.slug = slug;
            return tree;
        }
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    private static class Roles {

        @JsonProperty("slug")
        public String slug = "role";

        @JsonProperty("values")
        public Map<String, Role> values = new LinkedHashMap<>();
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    private static class Role {

        @JsonProperty("visibility")
        public Boolean visibility = true;

        @JsonProperty("name")
        public StringMultiLanguage name;

        @JsonProperty("help")
        public StringMultiLanguage help;

        public static Role of(StringMultiLanguage name, StringMultiLanguage help) {
            Role r = new Role();
            r.help = help;
            r.name = name;
            return r;
        }
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    private static class StringMultiLanguage {

        @JsonProperty("ru")
        public String ru;

        @JsonProperty("en")
        public String en;

        public static StringMultiLanguage of(String nameRu, String nameEn) {
            StringMultiLanguage result = new StringMultiLanguage();
            result.ru = nameRu;
            result.en = nameEn;
            return result;
        }
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class Field {

        @JsonProperty("slug")
        public String slug;

        @JsonProperty("name")
        public StringMultiLanguage name;

        @JsonProperty("type")
        public String type;

        @JsonProperty("required")
        public boolean required;

        public static Field of(String slug, String nameRu, String nameEn, String type, boolean required) {
            Field f = new Field();
            f.slug = slug;
            f.name = StringMultiLanguage.of(nameRu, nameEn);
            f.type = type;
            f.required = required;
            return f;
        }
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class RoleSubTree {

        @JsonIgnore
        public RoleDto.Type type;

        @JsonProperty("name")
        public StringMultiLanguage name;

        @JsonProperty("roles")
        public Roles roles = new Roles();

        @JsonProperty("fields")
        public List<Field> fields;

        public static RoleSubTree of(RoleDto.Type type, String nameRu, String nameEn) {
            RoleSubTree roleSubTree = new RoleSubTree();
            roleSubTree.type = type;
            roleSubTree.name = StringMultiLanguage.of(nameRu, nameEn);
            return roleSubTree;
        }

        public void addRole(String roleName, String nameRu, String nameEn) {
            roles.values.put(roleName, Role.of(StringMultiLanguage.of(nameRu, nameEn), null));
        }

        public void addRole(ru.yandex.solomon.auth.roles.Role role, String nameRu, String nameEn) {
            addRole(role, nameRu, nameEn, true);
        }

        public void addRole(ru.yandex.solomon.auth.roles.Role role, String nameRu, String nameEn, boolean visible) {
            String permissions = role.getPermissions().stream().map(Permission::getSlug).collect(Collectors.joining(", "));
            String ruHelp = helpPrefixRu(role) + "Набор разрешений: " + permissions;
            String enHelp = helpPrefixEn(role) + "Permissions list: " + permissions;
            Role r = Role.of(StringMultiLanguage.of(nameRu, nameEn), StringMultiLanguage.of(ruHelp, enHelp));
            r.visibility = visible;
            roles.values.put(role.name(), r);
        }

        private String helpPrefixRu(ru.yandex.solomon.auth.roles.Role role) {
            if (role == ru.yandex.solomon.auth.roles.Role.ADMIN) {
                return "Доступны все действия в системе мониторинга. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.PROJECT_ADMIN) {
                return "Может делать в проекте всё, включая добавление новых администраторов. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.JNS_SENDER) {
                return "Может отправлять события в JNS. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.PUSHER) {
                return "Может только поставлять метрики. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.EDITOR) {
                return "Может редактировать конфигурацию проекта. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.MUTER) {
                return "Может использовать мьюты. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.VIEWER) {
                return "Может читать конфигурацию проекта, а также метрики. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.SERVICE_PROVIDER_ADMIN ||
                    role == ru.yandex.solomon.auth.roles.Role.SERVICE_PROVIDER_ALERT_EDITOR  ) {
                return "Может редактировать алерты. ";
            }
            return "";
        }

        private String helpPrefixEn(ru.yandex.solomon.auth.roles.Role role) {
            if (role == ru.yandex.solomon.auth.roles.Role.ADMIN) {
                return "All actions. ";
            } else  if (role == ru.yandex.solomon.auth.roles.Role.PROJECT_ADMIN) {
                return "All action in project. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.JNS_SENDER) {
                return "Can send events to JNS. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.PUSHER) {
                return "Can push metrics. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.EDITOR) {
                return "Can edit project configuration. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.MUTER) {
                return "Can use mutes. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.VIEWER) {
                return "Can read project configuration and metrics. ";
            } else if (role == ru.yandex.solomon.auth.roles.Role.SERVICE_PROVIDER_ADMIN ||
                    role == ru.yandex.solomon.auth.roles.Role.SERVICE_PROVIDER_ALERT_EDITOR  ) {
                return "Can edit alerts. ";
            }
            return "";
        }
    }
}
