package ru.yandex.solomon.role;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.acl.db.memory.InMemoryGroupMemberDao;
import ru.yandex.solomon.acl.db.model.GroupMember;
import ru.yandex.solomon.locks.DistributedLockStub;
import ru.yandex.solomon.roles.GroupMembersUpdateScheduler;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;
import ru.yandex.solomon.util.host.HostUtils;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.solomon.role.StaffClientStub.GROUP_ID1;
import static ru.yandex.solomon.role.StaffClientStub.GROUP_ID2;
import static ru.yandex.solomon.role.StaffClientStub.USER_ID1;
import static ru.yandex.solomon.role.StaffClientStub.USER_ID2;
import static ru.yandex.solomon.role.StaffClientStub.USER_ID3;

/**
 * @author Alexey Trushkin
 */
public class GroupMembersUpdateSchedulerTest {

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(1, TimeUnit.MINUTES)
            .build();

    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private DistributedLockStub lock;
    private GroupMembersUpdateScheduler scheduler;
    private InMemoryGroupMemberDao dao;

    @Before
    public void setUp() throws Exception {
        dao = new InMemoryGroupMemberDao();
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);
        lock = new DistributedLockStub(clock);
        scheduler = new GroupMembersUpdateScheduler(
                dao,
                new StaffClientStub(),
                lock,
                ForkJoinPool.commonPool(),
                timer,
                clock,
                new MetricRegistry(),
                10000);
    }

    @After
    public void tearDown() {
        scheduler.close();
        timer.shutdownNow();
    }

    @Test
    public void updateMembers() {
        createSync(groupMember(USER_ID1, GROUP_ID1));
        createSync(groupMember(USER_ID1, GROUP_ID2));
        createSync(groupMember(USER_ID1 + "1", GROUP_ID2));
        createSync(groupMember(USER_ID1, GROUP_ID2 + "2"));

        Map<String, List<GroupMember>> initial = getAllSync();
        assertEquals(3, initial.size());
        assertEquals(1, initial.get(GROUP_ID1).size());
        assertTrue(initial.get(GROUP_ID1).contains(groupMember(USER_ID1, GROUP_ID1)));
        assertEquals(2, initial.get(GROUP_ID2).size());
        assertTrue(initial.get(GROUP_ID2).contains(groupMember(USER_ID1, GROUP_ID2)));
        assertTrue(initial.get(GROUP_ID2).contains(groupMember(USER_ID1 + "1", GROUP_ID2)));
        assertEquals(1, initial.get(GROUP_ID2 + "2").size());
        assertTrue(initial.get(GROUP_ID2 + "2").contains(groupMember(USER_ID1, GROUP_ID2 + "2")));

        lock.setOwner(HostUtils.getFqdn());
        join(scheduler.updateMembers());

        Map<String, List<GroupMember>> actual = getAllSync();
        assertEquals(2, actual.size());
        assertEquals(2, actual.get(GROUP_ID1).size());
        assertTrue(actual.get(GROUP_ID1).contains(groupMember(USER_ID1, GROUP_ID1)));
        assertTrue(actual.get(GROUP_ID1).contains(groupMember(USER_ID2, GROUP_ID1)));
        assertEquals(3, actual.get(GROUP_ID2).size());
        assertTrue(actual.get(GROUP_ID2).contains(groupMember(USER_ID1, GROUP_ID2)));
        assertTrue(actual.get(GROUP_ID2).contains(groupMember(USER_ID2, GROUP_ID2)));
        assertTrue(actual.get(GROUP_ID2).contains(groupMember(USER_ID3, GROUP_ID2)));
    }

    private Map<String, List<GroupMember>> getAllSync() {
        return join(dao.getAll().thenApply(groupMembers ->
                groupMembers.stream().collect(Collectors.groupingBy(GroupMember::groupId))));
    }

    private void createSync(GroupMember groupMember) {
        join(dao.create(List.of(groupMember)));
    }

    private GroupMember groupMember(String userId, String groupId) {
        return new GroupMember(groupId, userId);
    }
}
