package ru.yandex.solomon.role;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Before;
import org.junit.Test;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.roles.idm.IdmDtoConverter;
import ru.yandex.solomon.roles.idm.dto.IdmAddRoleDto;
import ru.yandex.solomon.roles.idm.dto.IdmOperationDto;
import ru.yandex.solomon.roles.idm.dto.IdmRemoveRoleDto;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class IdmDtoConverterTest {

    private static final String LOGIN = "test";
    private static final String GROUP = "group";
    private static final String UID = "uid 1";
    private static final String USER_SUBJECT = "user";
    private static final String TVM_SUBJECT = "tvm_app";
    private static final String PROJECT_ROLE = "{\"type\": \"project\", \"role\": \"admin\"}";
    private static final String SP_ROLE = "{\"type\": \"service_provider\", \"role\": \"admin\"}";
    private static final String SYSTEM_ROLE = "{\"type\": \"system\", \"role\": \"admin\"}";
    private static final String PROJECT_ROLE_FIELDS = "{\"project\": \"solomon\"}";
    private static final String SP_ROLE_FIELDS = "{\"serviceProvider\": \"solomon\"}";

    private IdmDtoConverter converter;

    @Before
    public void before() {
        converter = new IdmDtoConverter();
    }

    @Test
    public void newIdmAddRoleDto_projectRole() {
        MultiValueMap<String, String> data = new LinkedMultiValueMap<>();
        data.add("login", LOGIN);
        data.add("uid", UID);
        data.add("subject_type", USER_SUBJECT);
        data.add("role", PROJECT_ROLE);
        data.add("fields", PROJECT_ROLE_FIELDS);
        IdmAddRoleDto actual = converter.newIdmAddRoleDto(data);

        assertBaseProjectRole(actual);
    }

    @Test
    public void newIdmAddRoleDto_spRole() {
        MultiValueMap<String, String> data = new LinkedMultiValueMap<>();
        data.add("login", LOGIN);
        data.add("uid", UID);
        data.add("subject_type", USER_SUBJECT);
        data.add("role", SP_ROLE);
        data.add("fields", SP_ROLE_FIELDS);
        IdmAddRoleDto actual = converter.newIdmAddRoleDto(data);

        assertBaseSpRole(actual);
    }

    @Test
    public void newIdmAddRoleDto_systemRole() {
        MultiValueMap<String, String> data = new LinkedMultiValueMap<>();
        data.add("login", LOGIN);
        data.add("uid", UID);
        data.add("subject_type", USER_SUBJECT);
        data.add("role", SYSTEM_ROLE);
        IdmAddRoleDto actual = converter.newIdmAddRoleDto(data);

        assertBaseSystemRole(actual);
    }

    @Test
    public void newIdmAddRoleDto_systemRole_forGroup() {
        MultiValueMap<String, String> data = new LinkedMultiValueMap<>();
        data.add("group", GROUP);
        data.add("role", SYSTEM_ROLE);
        IdmAddRoleDto actual = converter.newIdmAddRoleDto(data);

        assertBaseSystemRole_group(actual);
    }

    @Test
    public void newIdmAddRoleDto_projectRole_forTVM() {
        MultiValueMap<String, String> data = new LinkedMultiValueMap<>();
        data.add("login", LOGIN);
        data.add("uid", UID);
        data.add("subject_type", TVM_SUBJECT);
        data.add("role", PROJECT_ROLE);
        data.add("fields", PROJECT_ROLE_FIELDS);
        IdmAddRoleDto actual = converter.newIdmAddRoleDto(data);

        assertBaseProjectRole_tvm(actual);
    }

    @Test
    public void newIdmRemoveRoleDto_projectRole() {
        MultiValueMap<String, String> data = new LinkedMultiValueMap<>();
        data.add("login", LOGIN);
        data.add("uid", UID);
        data.add("subject_type", USER_SUBJECT);
        data.add("role", PROJECT_ROLE);
        data.add("fields", PROJECT_ROLE_FIELDS);
        IdmRemoveRoleDto actual = converter.newIdmRemoveRoleDto(data);

        assertBaseProjectRole(actual);
        assertFalse(actual.delete);
    }

    @Test
    public void newIdmRemoveRoleDto_spRole() {
        MultiValueMap<String, String> data = new LinkedMultiValueMap<>();
        data.add("login", LOGIN);
        data.add("uid", UID);
        data.add("subject_type", USER_SUBJECT);
        data.add("role", SP_ROLE);
        data.add("fields", SP_ROLE_FIELDS);
        IdmRemoveRoleDto actual = converter.newIdmRemoveRoleDto(data);

        assertBaseSpRole(actual);
        assertFalse(actual.delete);
    }

    @Test
    public void newIdmRemoveRoleDto_systemRole() {
        MultiValueMap<String, String> data = new LinkedMultiValueMap<>();
        data.add("login", LOGIN);
        data.add("uid", UID);
        data.add("subject_type", USER_SUBJECT);
        data.add("role", SYSTEM_ROLE);
        data.add("fired", "1");
        IdmRemoveRoleDto actual = converter.newIdmRemoveRoleDto(data);

        assertBaseSystemRole(actual);
        assertTrue(actual.delete);
    }

    @Test
    public void newIdmRemoveRoleDto_systemRole_forGroup() {
        MultiValueMap<String, String> data = new LinkedMultiValueMap<>();
        data.add("group", GROUP);
        data.add("role", SYSTEM_ROLE);
        data.add("deleted", "1");
        IdmRemoveRoleDto actual = converter.newIdmRemoveRoleDto(data);

        assertBaseSystemRole_group(actual);
        assertTrue(actual.delete);
    }

    @Test
    public void newIdmRemoveRoleDto_projectRole_forTVM() {
        MultiValueMap<String, String> data = new LinkedMultiValueMap<>();
        data.add("login", LOGIN);
        data.add("uid", UID);
        data.add("subject_type", TVM_SUBJECT);
        data.add("role", PROJECT_ROLE);
        data.add("fields", PROJECT_ROLE_FIELDS);
        IdmRemoveRoleDto actual = converter.newIdmRemoveRoleDto(data);

        assertBaseProjectRole_tvm(actual);
    }

    private void assertBaseProjectRole(IdmOperationDto actual) {
        assertEquals(actual.login, LOGIN);
        assertEquals(actual.role.roleId, "admin");
        assertEquals(actual.aclUidType, AclUidType.USER);
        assertEquals(actual.project, "solomon");
        assertTrue(actual.role.isProject());
        assertFalse(actual.role.isServiceProvider());
        assertFalse(actual.role.isSystem());
        assertTrue(actual.isUser());
        assertFalse(actual.isGroup());
        assertFalse(actual.isTvm());
    }

    private void assertBaseSpRole(IdmOperationDto actual) {
        assertEquals(actual.login, LOGIN);
        assertEquals(actual.role.roleId, "admin");
        assertEquals(actual.aclUidType, AclUidType.USER);
        assertEquals(actual.serviceProviderId, "solomon");
        assertFalse(actual.role.isProject());
        assertTrue(actual.role.isServiceProvider());
        assertFalse(actual.role.isSystem());
        assertTrue(actual.isUser());
        assertFalse(actual.isGroup());
        assertFalse(actual.isTvm());
    }

    private void assertBaseSystemRole(IdmOperationDto actual) {
        assertEquals(actual.login, LOGIN);
        assertEquals(actual.role.roleId, "admin");
        assertEquals(actual.aclUidType, AclUidType.USER);
        assertFalse(actual.role.isProject());
        assertTrue(actual.role.isSystem());
        assertFalse(actual.role.isServiceProvider());
        assertTrue(actual.isUser());
        assertFalse(actual.isGroup());
        assertFalse(actual.isTvm());
    }

    private void assertBaseSystemRole_group(IdmOperationDto actual) {
        assertEquals(actual.group, GROUP);
        assertEquals(actual.role.roleId, "admin");
        assertEquals(actual.aclUidType, AclUidType.GROUP);
        assertFalse(actual.role.isProject());
        assertTrue(actual.role.isSystem());
        assertFalse(actual.role.isServiceProvider());
        assertFalse(actual.isUser());
        assertTrue(actual.isGroup());
        assertFalse(actual.isTvm());
    }

    private void assertBaseProjectRole_tvm(IdmOperationDto actual) {
        assertEquals(actual.login, LOGIN);
        assertEquals(actual.role.roleId, "admin");
        assertEquals(actual.aclUidType, AclUidType.TVM);
        assertEquals(actual.project, "solomon");
        assertTrue(actual.role.isProject());
        assertFalse(actual.role.isSystem());
        assertFalse(actual.role.isServiceProvider());
        assertFalse(actual.isUser());
        assertFalse(actual.isGroup());
        assertTrue(actual.isTvm());
    }
}
