package ru.yandex.solomon.role;

import java.util.List;
import java.util.Set;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.idm.dto.RoleRequestDto;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.config.protobuf.TIdmConfig;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.roles.idm.IdmMigrationService;
import ru.yandex.solomon.roles.idm.dto.IdmMigrateProjectsDto;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * @author Alexey Trushkin
 */
public class IdmMigrationServiceTest {

    protected static final String PROJECT = "solomon";
    private static final String PROJECT2 = "solomon_cloud";
    protected static final String OWNER1 = "robot-solomon 1";
    private static final String OWNER2 = "robot-solomon 2";
    private static final String VIEWER_USER1 = "user 1";
    private static final String ADMIN_USER2 = "user 2";
    private static final String ADMIN_USER3 = "user 3";
    private static final String VIEWER_PUSHER_USER4 = "user 4";
    protected static final String SYSTEM = "system";
    private IdmMigrationService service;
    private StubIdmClient stubClient;

    @Before
    public void setUp() throws Exception {
        SolomonConfHolder holder = new SolomonConfHolder();
        Project project = Project.newBuilder()
                .setId(PROJECT)
                .setName(PROJECT)
                .setOwner(OWNER1)
                .setAcl(Acl.of(Set.of(VIEWER_USER1), Set.of(ADMIN_USER2), Set.of(ADMIN_USER3), Set.of(VIEWER_PUSHER_USER4)))
                .build();
        Project project2 = Project.newBuilder()
                .setId(PROJECT2)
                .setName(PROJECT2)
                .setOwner(OWNER2)
                .setAcl(Acl.of(Set.of(VIEWER_USER1), Set.of("tvm-1"), Set.of(), Set.of()))
                .build();
        holder.onConfigurationLoad(SolomonConfWithContext.create(new SolomonRawConf(
                List.of(),
                List.of(project, project2),
                List.of(),
                List.of(),
                List.of()
        )));
        TIdmConfig config = TIdmConfig.newBuilder()
                .setSystemName(SYSTEM)
                .build();
        service = new IdmMigrationService(
                stubClient = new StubIdmClient(),
                holder,
                config
        );
    }

    @Test
    public void migrate() {
        IdmMigrateProjectsDto dto = new IdmMigrateProjectsDto();
        dto.projects.add(PROJECT);
        dto.projects.add(PROJECT2);
        dto.projects.add("none");

        service.migrate(dto).join();

        List<RoleRequestDto> state = stubClient.getState();
        assertEquals(9, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(OWNER1, PROJECT, Role.PROJECT_ADMIN.name(), SYSTEM, false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(VIEWER_USER1, PROJECT, Role.VIEWER.name(), SYSTEM, false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(ADMIN_USER2, PROJECT, Role.PROJECT_ADMIN.name(), SYSTEM, false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(ADMIN_USER3, PROJECT, Role.PROJECT_ADMIN.name(), SYSTEM, false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(VIEWER_PUSHER_USER4, PROJECT, Role.VIEWER.name(), SYSTEM, false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(VIEWER_PUSHER_USER4, PROJECT, Role.PUSHER.name(), SYSTEM, false)));

        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(OWNER2, PROJECT2, Role.PROJECT_ADMIN.name(), SYSTEM, false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(VIEWER_USER1, PROJECT2, Role.VIEWER.name(), SYSTEM, false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("1", PROJECT2, Role.PROJECT_ADMIN.name(), SYSTEM, true)));
    }

    @Test
    public void delete() {
        service.migrateDeletedProjects(List.of(PROJECT, PROJECT2)).join();

        List<RoleRequestDto> state = stubClient.getDeleteRolesState();
        assertEquals(2, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(null, PROJECT, null, SYSTEM, false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(null, PROJECT2, null, SYSTEM, false)));
    }
}
