package ru.yandex.solomon.role;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionException;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.acl.db.memory.InMemoryProjectAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ProjectAclEntry;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.dao.memory.InMemoryProjectsDao;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.roles.ProjectRoleManager;
import ru.yandex.solomon.roles.idm.dto.IdmResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRolesPageResponseDto;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.role.DtoFactory.GROUP;
import static ru.yandex.solomon.role.DtoFactory.ROLE_ID;
import static ru.yandex.solomon.role.DtoFactory.ROLE_ID2;
import static ru.yandex.solomon.role.DtoFactory.ROLE_ID3;
import static ru.yandex.solomon.role.DtoFactory.UID;
import static ru.yandex.solomon.role.DtoFactory.USER;
import static ru.yandex.solomon.role.DtoFactory.idmAddRoleDto;
import static ru.yandex.solomon.role.DtoFactory.idmAddUserRoleDto;
import static ru.yandex.solomon.role.DtoFactory.idmRemoveRoleDto;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ProjectRoleManagerTest {

    private ProjectRoleManager manager;
    private InMemoryProjectAclEntryDao dao;
    private InMemoryProjectsDao projectsDao;

    @Before
    public void before() {
        dao = new InMemoryProjectAclEntryDao();
        projectsDao = new InMemoryProjectsDao();
        SolomonConfHolder holder = new SolomonConfHolder();
        Project project = Project.newBuilder()
                .setId(UID)
                .setName(UID)
                .setOwner("robot-solomon")
                .build();
        holder.onConfigurationLoad(SolomonConfWithContext.create(new SolomonRawConf(
                List.of(),
                List.of(project),
                List.of(),
                List.of(),
                List.of()
        )));
        manager = new ProjectRoleManager(dao, holder, projectsDao);
    }

    @Test
    public void addRole() {
        IdmResponseDto.ResultData data = manager.addRole(idmAddRoleDto(ROLE_ID)).join();

        Optional<ProjectAclEntry> actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();

        assertEquals(UID, data.project);
        assertEntry(actualOptional, Set.of(ROLE_ID), UID);
    }

    @Test
    public void addRole_unknownProject() {
        try {
            manager.addRole(idmAddRoleDto(ROLE_ID, "someProject", null)).join();
        } catch (CompletionException e) {
            assertEquals("Hasn't project someProject", e.getCause().getMessage());
            return;
        }
        throw new RuntimeException("Can't reach");
    }

    @Test
    public void addRole_createdProject() {
        projectsDao.insert(Project.newBuilder()
                .setId("notLoaded")
                .setName("notLoaded")
                .setOwner("solomon")
                .build())
                .join();
        IdmResponseDto.ResultData data = manager.addRole(idmAddRoleDto(ROLE_ID, "notLoaded", null)).join();

        Optional<ProjectAclEntry> actualOptional = dao.find("notLoaded", GROUP, AclUidType.GROUP).join();

        assertEquals("notLoaded", data.project);
        assertEntry(actualOptional, Set.of(ROLE_ID), "notLoaded");
    }

    @Test
    public void addSomeRoles() {
        manager.addRole(idmAddRoleDto(ROLE_ID)).join();
        manager.addRole(idmAddRoleDto(ROLE_ID2)).join();

        Optional<ProjectAclEntry> actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();

        assertEntry(actualOptional, Set.of(ROLE_ID, ROLE_ID2), UID);
    }

    @Test
    public void removeRole() {
        manager.addRole(idmAddRoleDto(ROLE_ID)).join();
        manager.addRole(idmAddRoleDto(ROLE_ID2)).join();


        Optional<ProjectAclEntry> actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();
        assertEntry(actualOptional, Set.of(ROLE_ID, ROLE_ID2), UID);

        manager.removeRole(idmRemoveRoleDto(ROLE_ID2)).join();

        actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();
        assertEntry(actualOptional, Set.of(ROLE_ID), UID);

        manager.removeRole(idmRemoveRoleDto(ROLE_ID)).join();

        actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();
        assertFalse(actualOptional.isPresent());
    }

    @Test
    public void removeUnknown() {
        manager.addRole(idmAddRoleDto(ROLE_ID)).join();
        manager.removeRole(idmRemoveRoleDto(ROLE_ID2)).join();

        Optional<ProjectAclEntry> actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();
        assertEntry(actualOptional, Set.of(ROLE_ID), UID);
    }

    @Test
    public void getRoles() {
        manager.addRole(idmAddRoleDto(ROLE_ID)).join();
        manager.addRole(idmAddRoleDto(ROLE_ID2)).join();
        manager.addRole(idmAddUserRoleDto(ROLE_ID3)).join();

        List<IdmRolesPageResponseDto.Role> roles = manager.getRoles().join();

        assertEquals(3, roles.size());
        var role1 = roles.stream().filter(role -> role.path.endsWith(ROLE_ID + "/")).findFirst().get();
        assertEquals(Integer.valueOf(GROUP), role1.group);
        assertEquals("/type/project/role/" + ROLE_ID + "/", role1.path);
        assertEquals(Map.of("project", UID), role1.fields);

        var role2 = roles.stream().filter(role -> role.path.endsWith(ROLE_ID2 + "/")).findFirst().get();
        assertEquals(Integer.valueOf(GROUP), role2.group);
        assertEquals("/type/project/role/" + ROLE_ID2 + "/", role2.path);
        assertEquals(Map.of("project", UID), role2.fields);

        var role3 = roles.stream().filter(role -> role.path.endsWith(ROLE_ID3 + "/")).findFirst().get();
        assertEquals(USER, role3.login);
        assertEquals("user", role3.subjectType);
        assertEquals("/type/project/role/" + ROLE_ID3 + "/", role3.path);
        assertEquals(Map.of("project", UID), role3.fields);
    }


    private void assertEntry(Optional<ProjectAclEntry> actualOptional, Set<String> roles, String project) {
        assertTrue(actualOptional.isPresent());
        ProjectAclEntry entry = actualOptional.get();
        assertEquals(entry.getProjectId(), project);
        assertEquals(entry.getUid(), GROUP);
        assertEquals(entry.getType(), AclUidType.GROUP);
        assertEquals(entry.getRoles(), roles);
    }
}
