package ru.yandex.solomon.role;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.idm.dto.RoleRequestDto;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.config.protobuf.TIdmConfig;
import ru.yandex.solomon.core.db.dao.memory.InMemoryProjectsDao;
import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.roles.RoleAclSynchronizer;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.role.DtoFactory.UID;
import static ru.yandex.solomon.role.DtoFactory.USER;
import static ru.yandex.solomon.role.DtoFactory.idmAddRoleDto;
import static ru.yandex.solomon.role.DtoFactory.idmAddSystemRoleDto;
import static ru.yandex.solomon.role.DtoFactory.idmAddUserRoleDto;
import static ru.yandex.solomon.role.DtoFactory.idmRemoveRoleDto;
import static ru.yandex.solomon.role.DtoFactory.idmRemoveSystemRoleDto;
import static ru.yandex.solomon.role.DtoFactory.idmRemoveUserRoleDto;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class RoleAclSynchronizerTest {

    private RoleAclSynchronizer roleAclSynchronizer;
    private InMemoryProjectsDao projectsDao;
    private StubIdmClient client;

    @Before
    public void before() {
        projectsDao = new InMemoryProjectsDao();
        TIdmConfig config = TIdmConfig.newBuilder()
                .setSystemName("sys")
                .build();
        client = new StubIdmClient();
        roleAclSynchronizer = new RoleAclSynchronizer(projectsDao, config, client);
    }

    @Test
    public void roleAdded() {
        roleAclSynchronizer.roleAdded(idmAddRoleDto("")).join();
        roleAclSynchronizer.roleAdded(idmAddSystemRoleDto("")).join();
        addProject();

        roleAclSynchronizer.roleAdded(idmAddUserRoleDto(Role.PROJECT_ADMIN.name())).join();
        roleAclSynchronizer.roleAdded(idmAddUserRoleDto(Role.PROJECT_ADMIN.name(), "user1")).join();
        roleAclSynchronizer.roleAdded(idmAddUserRoleDto(Role.EDITOR.name())).join();
        roleAclSynchronizer.roleAdded(idmAddUserRoleDto(Role.EDITOR.name(), "user2")).join();
        roleAclSynchronizer.roleAdded(idmAddUserRoleDto(Role.VIEWER.name())).join();
        roleAclSynchronizer.roleAdded(idmAddUserRoleDto(Role.VIEWER.name(), "user3")).join();
        roleAclSynchronizer.roleAdded(idmAddUserRoleDto(Role.PUSHER.name())).join();
        roleAclSynchronizer.roleAdded(idmAddUserRoleDto(Role.PUSHER.name(), "user4")).join();

        Optional<Project> aNew = projectsDao.findById(UID).join();

        assertTrue(aNew.isPresent());
        assertEquals(2, aNew.get().getAcl().getCanDelete().size());
        assertTrue(aNew.get().getAcl().getCanDelete().contains(USER));
        assertTrue(aNew.get().getAcl().getCanDelete().contains("user1"));
        assertEquals(2, aNew.get().getAcl().getCanUpdate().size());
        assertTrue(aNew.get().getAcl().getCanUpdate().contains(USER));
        assertTrue(aNew.get().getAcl().getCanUpdate().contains("user1"));
        assertEquals(2, aNew.get().getAcl().getCanRead().size());
        assertTrue(aNew.get().getAcl().getCanRead().contains(USER));
        assertTrue(aNew.get().getAcl().getCanRead().contains("user3"));
        assertEquals(2, aNew.get().getAcl().getCanWrite().size());
        assertTrue(aNew.get().getAcl().getCanWrite().contains(USER));
        assertTrue(aNew.get().getAcl().getCanWrite().contains("user4"));
    }

    @Test
    public void roleRemoved() {
        roleAclSynchronizer.roleRemoved(idmRemoveRoleDto("")).join();
        roleAclSynchronizer.roleRemoved(idmRemoveSystemRoleDto("")).join();

        addProjectWithAcl();

        roleAclSynchronizer.roleRemoved(idmRemoveUserRoleDto(Role.PROJECT_ADMIN.name())).join();
        roleAclSynchronizer.roleRemoved(idmRemoveUserRoleDto(Role.EDITOR.name())).join();
        roleAclSynchronizer.roleRemoved(idmRemoveUserRoleDto(Role.VIEWER.name())).join();
        roleAclSynchronizer.roleRemoved(idmRemoveUserRoleDto(Role.PUSHER.name())).join();

        Optional<Project> aNew = projectsDao.findById(UID).join();
        assertTrue(aNew.isPresent());
        assertEquals(1, aNew.get().getAcl().getCanDelete().size());
        assertEquals(1, aNew.get().getAcl().getCanUpdate().size());
        assertEquals(1, aNew.get().getAcl().getCanRead().size());
        assertEquals(1, aNew.get().getAcl().getCanWrite().size());
        assertTrue(aNew.get().getAcl().getCanWrite().contains("user4"));
        assertTrue(aNew.get().getAcl().getCanDelete().contains("user1"));
        assertTrue(aNew.get().getAcl().getCanUpdate().contains("user1"));
        assertTrue(aNew.get().getAcl().getCanRead().contains("user3"));

        roleAclSynchronizer.roleRemoved(idmRemoveUserRoleDto(Role.EDITOR.name(), "user2")).join();
        roleAclSynchronizer.roleRemoved(idmRemoveUserRoleDto(Role.VIEWER.name(), "user3")).join();
        roleAclSynchronizer.roleRemoved(idmRemoveUserRoleDto(Role.PROJECT_ADMIN.name(), "user1")).join();
        roleAclSynchronizer.roleRemoved(idmRemoveUserRoleDto(Role.PUSHER.name(), "user4")).join();

        aNew = projectsDao.findById(UID).join();
        assertTrue(aNew.isPresent());
        assertEquals(0, aNew.get().getAcl().getCanDelete().size());
        assertEquals(0, aNew.get().getAcl().getCanUpdate().size());
        assertEquals(0, aNew.get().getAcl().getCanRead().size());
        assertEquals(0, aNew.get().getAcl().getCanWrite().size());
    }

    @Test
    public void changedUpdateDelete() {
        // No changes
        roleAclSynchronizer.changed(
                getProject(Acl.empty()),
                getProject(Acl.empty())
        ).join();

        List<RoleRequestDto> state = client.getState();
        assertEquals(0, state.size());

        // Add 3 update
        roleAclSynchronizer.changed(
                getProject(Acl.empty()),
                getProject(Acl.of(Set.of(), Set.of("user1", USER, "tvm-1"), Set.of(), Set.of()))
        ).join();

        state = client.getState();
        assertEquals(3, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("user1", UID, Role.PROJECT_ADMIN.name(), "sys", false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("1", UID, Role.PROJECT_ADMIN.name(), "sys", true)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(USER, UID, Role.PROJECT_ADMIN.name(), "sys", false)));

        // Add 2 delete
        roleAclSynchronizer.changed(
                getProject(Acl.empty()),
                getProject(Acl.of(Set.of(), Set.of(), Set.of("user11", "tvm-12"), Set.of()))
        ).join();

        state = client.getState();
        assertEquals(5, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("user11", UID, Role.PROJECT_ADMIN.name(), "sys", false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("12", UID, Role.PROJECT_ADMIN.name(), "sys", true)));

        // Add 3 update and 3 delete, but 2 of them the same
        roleAclSynchronizer.changed(
                getProject(Acl.empty()),
                getProject(Acl.of(Set.of(), Set.of("user111", "tvm-123", "user4"), Set.of("user111", "tvm-123", "user5"), Set.of()))
        ).join();

        state = client.getState();
        assertEquals(9, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("user111", UID, Role.PROJECT_ADMIN.name(), "sys", false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("123", UID, Role.PROJECT_ADMIN.name(), "sys", true)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("user4", UID, Role.PROJECT_ADMIN.name(), "sys", false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("user5", UID, Role.PROJECT_ADMIN.name(), "sys", false)));

        // Transfer from delete to update, no changes
        roleAclSynchronizer.changed(
                getProject(Acl.of(Set.of(), Set.of(), Set.of("user2"), Set.of())),
                getProject(Acl.of(Set.of(), Set.of("user2"), Set.of(), Set.of()))
        ).join();

        state = client.getState();
        assertEquals(9, state.size());
        var deleteState = client.getDeleteState();
        Assert.assertEquals(0, deleteState.size());

        // Delete 2 and add 2
        roleAclSynchronizer.changed(
                getProject(Acl.of(Set.of(), Set.of("userOld"), Set.of("user2", "userOld2"), Set.of())),
                getProject(Acl.of(Set.of(), Set.of("user2", "userNew2"), Set.of("userNew"), Set.of()))
        ).join();

        state = client.getState();
        assertEquals(11, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("userNew", UID, Role.PROJECT_ADMIN.name(), "sys", false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("userNew2", UID, Role.PROJECT_ADMIN.name(), "sys", false)));

        deleteState = client.getDeleteState();
        Assert.assertEquals(2, deleteState.size());
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld", UID, Role.PROJECT_ADMIN.name(), "sys", false)));
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld2", UID, Role.PROJECT_ADMIN.name(), "sys", false)));

        // Delete all
        roleAclSynchronizer.changed(
                getProject(Acl.of(Set.of(), Set.of("userOld3"), Set.of("userOld5", "userOld4"), Set.of())),
                getProject(Acl.of(Set.of(), Set.of(), Set.of(), Set.of()))
        ).join();

        state = client.getState();
        assertEquals(11, state.size());

        deleteState = client.getDeleteState();
        Assert.assertEquals(5, deleteState.size());
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld3", UID, Role.PROJECT_ADMIN.name(), "sys", false)));
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld4", UID, Role.PROJECT_ADMIN.name(), "sys", false)));
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld5", UID, Role.PROJECT_ADMIN.name(), "sys", false)));
    }

    @Test
    public void changedRead() {
        // No changes
        roleAclSynchronizer.changed(
                getProject(Acl.empty()),
                getProject(Acl.empty())
        ).join();

        List<RoleRequestDto> state = client.getState();
        assertEquals(0, state.size());

        // Add 3
        roleAclSynchronizer.changed(
                getProject(Acl.empty()),
                getProject(Acl.of(Set.of("user1", USER, "tvm-1"), Set.of(), Set.of(), Set.of()))
        ).join();

        state = client.getState();
        assertEquals(3, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("user1", UID, Role.VIEWER.name(), "sys", false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("1", UID, Role.VIEWER.name(), "sys", true)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(USER, UID, Role.VIEWER.name(), "sys", false)));

        // Add 1 new
        roleAclSynchronizer.changed(
                getProject(Acl.of(Set.of("user1", USER, "tvm-1"), Set.of(), Set.of(), Set.of())),
                getProject(Acl.of(Set.of("user1", USER, "tvm-1", "user2"), Set.of(), Set.of(), Set.of()))
        ).join();

        state = client.getState();
        assertEquals(4, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("user2", UID, Role.VIEWER.name(), "sys", false)));

        // Delete 2 and add 2
        roleAclSynchronizer.changed(
                getProject(Acl.of(Set.of("userOld", USER, "tvm-1"), Set.of(), Set.of(), Set.of())),
                getProject(Acl.of(Set.of("user3", USER, "tvm-12"), Set.of(), Set.of(), Set.of()))
        ).join();

        state = client.getState();
        assertEquals(6, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("user3", UID, Role.VIEWER.name(), "sys", false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("12", UID, Role.VIEWER.name(), "sys", true)));

        var deleteState = client.getDeleteState();
        Assert.assertEquals(2, deleteState.size());
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld", UID, Role.VIEWER.name(), "sys", false)));
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("1", UID, Role.VIEWER.name(), "sys", true)));

        // Delete all
        roleAclSynchronizer.changed(
                getProject(Acl.of(Set.of("userOld3", "userOld5", "userOld4"), Set.of(), Set.of(), Set.of())),
                getProject(Acl.of(Set.of(), Set.of(), Set.of(), Set.of()))
        ).join();

        state = client.getState();
        assertEquals(6, state.size());

        deleteState = client.getDeleteState();
        Assert.assertEquals(5, deleteState.size());
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld3", UID, Role.VIEWER.name(), "sys", false)));
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld4", UID, Role.VIEWER.name(), "sys", false)));
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld5", UID, Role.VIEWER.name(), "sys", false)));
    }

    @Test
    public void changedWrite() {
        // No changes
        roleAclSynchronizer.changed(
                getProject(Acl.empty()),
                getProject(Acl.empty())
        ).join();

        List<RoleRequestDto> state = client.getState();
        assertEquals(0, state.size());

        // Add 3
        roleAclSynchronizer.changed(
                getProject(Acl.empty()),
                getProject(Acl.of(Set.of(), Set.of(), Set.of(), Set.of("user1", USER, "tvm-1")))
        ).join();

        state = client.getState();
        assertEquals(3, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("user1", UID, Role.PUSHER.name(), "sys", false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("1", UID, Role.PUSHER.name(), "sys", true)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest(USER, UID, Role.PUSHER.name(), "sys", false)));

        // Add 1 new
        roleAclSynchronizer.changed(
                getProject(Acl.of(Set.of(), Set.of(), Set.of(), Set.of("user1", USER, "tvm-1"))),
                getProject(Acl.of(Set.of(), Set.of(), Set.of(), Set.of("user1", USER, "tvm-1", "user2")))
        ).join();

        state = client.getState();
        assertEquals(4, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("user2", UID, Role.PUSHER.name(), "sys", false)));

        // Delete 2 and add 2
        roleAclSynchronizer.changed(
                getProject(Acl.of(Set.of(), Set.of(), Set.of(), Set.of("userOld", USER, "tvm-1"))),
                getProject(Acl.of(Set.of(), Set.of(), Set.of(), Set.of("user3", USER, "tvm-12")))
        ).join();

        state = client.getState();
        assertEquals(6, state.size());
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("user3", UID, Role.PUSHER.name(), "sys", false)));
        assertTrue(state.contains(RoleRequestDto.newProjectRoleRequest("12", UID, Role.PUSHER.name(), "sys", true)));

        var deleteState = client.getDeleteState();
        Assert.assertEquals(2, deleteState.size());
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld", UID, Role.PUSHER.name(), "sys", false)));
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("1", UID, Role.PUSHER.name(), "sys", true)));

        // Delete all
        roleAclSynchronizer.changed(
                getProject(Acl.of(Set.of(), Set.of(), Set.of(), Set.of("userOld3", "userOld5", "userOld4"))),
                getProject(Acl.of(Set.of(), Set.of(), Set.of(), Set.of()))
        ).join();

        state = client.getState();
        assertEquals(6, state.size());

        deleteState = client.getDeleteState();
        Assert.assertEquals(5, deleteState.size());
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld3", UID, Role.PUSHER.name(), "sys", false)));
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld4", UID, Role.PUSHER.name(), "sys", false)));
        assertTrue(deleteState.contains(RoleRequestDto.newProjectRoleRequest("userOld5", UID, Role.PUSHER.name(), "sys", false)));
    }

    private Project getProject(Acl acl) {
        return Project.newBuilder()
                .setId(UID)
                .setName("new")
                .setOwner("my")
                .setAcl(acl)
                .build();
    }

    private void addProject() {
        projectsDao.insert(Project.newBuilder()
                .setId(UID)
                .setName("new")
                .setOwner("my")
                .build())
                .join();
    }

    private void addProjectWithAcl() {
        projectsDao.insert(Project.newBuilder()
                .setId(UID)
                .setName("new")
                .setOwner("my")
                .setAcl(Acl.of(Set.of("user3", USER), Set.of("user1", USER), Set.of("user1", USER), Set.of("user4", USER)))
                .build())
                .join();
    }
}
