package ru.yandex.solomon.role;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionException;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.acl.db.memory.InMemoryServiceProviderAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ServiceProviderAclEntry;
import ru.yandex.solomon.core.db.dao.memory.InMemoryYdbServiceProvidersDao;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.roles.ServiceProviderRoleManager;
import ru.yandex.solomon.roles.idm.dto.IdmResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRolesPageResponseDto;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.role.DtoFactory.GROUP;
import static ru.yandex.solomon.role.DtoFactory.ROLE_ID;
import static ru.yandex.solomon.role.DtoFactory.ROLE_ID2;
import static ru.yandex.solomon.role.DtoFactory.UID;
import static ru.yandex.solomon.role.DtoFactory.idmAddRoleDto;
import static ru.yandex.solomon.role.DtoFactory.idmRemoveRoleDto;
import static ru.yandex.solomon.role.DtoFactory.idmRemoveRoleDtoSp;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ServiceProviderRoleManagerTest {

    private ServiceProviderRoleManager manager;
    private InMemoryServiceProviderAclEntryDao dao;
    private InMemoryYdbServiceProvidersDao serviceProvidersDao;

    @Before
    public void before() {
        dao = new InMemoryServiceProviderAclEntryDao();
        serviceProvidersDao = new InMemoryYdbServiceProvidersDao();
        manager = new ServiceProviderRoleManager(dao, serviceProvidersDao);
        serviceProvidersDao.insert(ServiceProvider.newBuilder()
                .setId("some id 1")
                .build())
                .join();
    }

    @Test
    public void addRole() {
        IdmResponseDto.ResultData data = manager.addRole(idmAddRoleDto(ROLE_ID, null, UID)).join();

        Optional<ServiceProviderAclEntry> actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();

        assertEquals(UID, data.serviceProvider);
        assertEntry(actualOptional, Set.of(ROLE_ID), UID);
    }

    @Test
    public void addRole_unknownSP() {
        serviceProvidersDao.delete("some id 1").join();
        try {
            manager.addRole(idmAddRoleDto(ROLE_ID, null, "some sp")).join();
        } catch (CompletionException e) {
            assertEquals("Hasn't service provider some sp", e.getCause().getMessage());
            return;
        }
        throw new RuntimeException("Can't reach");
    }

    @Test
    public void addRole_createdSP() {
        IdmResponseDto.ResultData data = manager.addRole(idmAddRoleDto(ROLE_ID, null, "some id 1")).join();

        Optional<ServiceProviderAclEntry> actualOptional = dao.find("some id 1", GROUP, AclUidType.GROUP).join();

        assertEquals("some id 1", data.serviceProvider);
        assertEntry(actualOptional, Set.of(ROLE_ID), "some id 1");
    }

    @Test
    public void addSomeRoles() {
        manager.addRole(idmAddRoleDto(ROLE_ID, null, UID)).join();
        manager.addRole(idmAddRoleDto(ROLE_ID2, null, UID)).join();

        Optional<ServiceProviderAclEntry> actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();

        assertEntry(actualOptional, Set.of(ROLE_ID, ROLE_ID2), UID);
    }

    @Test
    public void removeRole() {
        manager.addRole(idmAddRoleDto(ROLE_ID, null, UID)).join();
        manager.addRole(idmAddRoleDto(ROLE_ID2, null, UID)).join();


        Optional<ServiceProviderAclEntry> actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();
        assertEntry(actualOptional, Set.of(ROLE_ID, ROLE_ID2), UID);

        manager.removeRole(idmRemoveRoleDtoSp(ROLE_ID2)).join();

        actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();
        assertEntry(actualOptional, Set.of(ROLE_ID), UID);

        manager.removeRole(idmRemoveRoleDtoSp(ROLE_ID)).join();

        actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();
        assertFalse(actualOptional.isPresent());
    }

    @Test
    public void removeUnknown() {
        manager.addRole(idmAddRoleDto(ROLE_ID, null, UID)).join();
        manager.removeRole(idmRemoveRoleDto(ROLE_ID2)).join();

        Optional<ServiceProviderAclEntry> actualOptional = dao.find(UID, GROUP, AclUidType.GROUP).join();
        assertEntry(actualOptional, Set.of(ROLE_ID), UID);
    }

    @Test
    public void getRoles() {
        manager.addRole(idmAddRoleDto(ROLE_ID, null, UID)).join();
        manager.addRole(idmAddRoleDto(ROLE_ID2, null, UID)).join();

        List<IdmRolesPageResponseDto.Role> roles = manager.getRoles().join();

        assertEquals(2, roles.size());
        var role1 = roles.stream().filter(role -> role.path.endsWith(ROLE_ID + "/")).findFirst().get();
        assertEquals(Integer.valueOf(GROUP), role1.group);
        assertEquals("/type/service_provider/role/" + ROLE_ID + "/", role1.path);
        assertEquals(Map.of("serviceProvider", UID), role1.fields);

        var role2 = roles.stream().filter(role -> role.path.endsWith(ROLE_ID2 + "/")).findFirst().get();
        assertEquals(Integer.valueOf(GROUP), role2.group);
        assertEquals("/type/service_provider/role/" + ROLE_ID2 + "/", role2.path);
        assertEquals(Map.of("serviceProvider", UID), role2.fields);
    }


    private void assertEntry(Optional<ServiceProviderAclEntry> actualOptional, Set<String> roles, String sp) {
        assertTrue(actualOptional.isPresent());
        ServiceProviderAclEntry entry = actualOptional.get();
        assertEquals(entry.getServiceProviderId(), sp);
        assertEquals(entry.getUid(), GROUP);
        assertEquals(entry.getType(), AclUidType.GROUP);
        assertEquals(entry.getRoles(), roles);
    }
}
