package ru.yandex.solomon.role;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.acl.db.memory.InMemorySystemAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.SystemAclEntry;
import ru.yandex.solomon.roles.SystemRoleManager;
import ru.yandex.solomon.roles.idm.dto.IdmRolesPageResponseDto;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.role.DtoFactory.GROUP;
import static ru.yandex.solomon.role.DtoFactory.ROLE_ID;
import static ru.yandex.solomon.role.DtoFactory.ROLE_ID2;
import static ru.yandex.solomon.role.DtoFactory.ROLE_ID3;
import static ru.yandex.solomon.role.DtoFactory.USER;
import static ru.yandex.solomon.role.DtoFactory.idmAddRoleDto;
import static ru.yandex.solomon.role.DtoFactory.idmAddUserRoleDto;
import static ru.yandex.solomon.role.DtoFactory.idmRemoveRoleDto;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class SystemRoleManagerTest {

    private SystemRoleManager manager;
    private InMemorySystemAclEntryDao dao;

    @Before
    public void before() {
        dao = new InMemorySystemAclEntryDao();
        manager = new SystemRoleManager(dao);
    }

    @Test
    public void addRole() {
        manager.addRole(idmAddRoleDto(ROLE_ID)).join();

        Optional<SystemAclEntry> actualOptional = dao.find(GROUP, AclUidType.GROUP).join();

        assertEntry(actualOptional, Set.of(ROLE_ID));
    }

    @Test
    public void addSomeRoles() {
        manager.addRole(idmAddRoleDto(ROLE_ID)).join();
        manager.addRole(idmAddRoleDto(ROLE_ID2)).join();

        Optional<SystemAclEntry> actualOptional = dao.find(GROUP, AclUidType.GROUP).join();

        assertEntry(actualOptional, Set.of(ROLE_ID, ROLE_ID2));
    }

    @Test
    public void removeRole() {
        manager.addRole(idmAddRoleDto(ROLE_ID)).join();
        manager.addRole(idmAddRoleDto(ROLE_ID2)).join();

        Optional<SystemAclEntry> actualOptional = dao.find(GROUP, AclUidType.GROUP).join();
        assertEntry(actualOptional, Set.of(ROLE_ID, ROLE_ID2));

        manager.removeRole(idmRemoveRoleDto(ROLE_ID2)).join();

        actualOptional = dao.find(GROUP, AclUidType.GROUP).join();
        assertEntry(actualOptional, Set.of(ROLE_ID));

        manager.removeRole(idmRemoveRoleDto(ROLE_ID)).join();

        actualOptional = dao.find(GROUP, AclUidType.GROUP).join();
        assertFalse(actualOptional.isPresent());
    }

    @Test
    public void removeUnknown() {
        manager.addRole(idmAddRoleDto(ROLE_ID)).join();
        manager.removeRole(idmRemoveRoleDto(ROLE_ID2)).join();

        Optional<SystemAclEntry> actualOptional = dao.find(GROUP, AclUidType.GROUP).join();
        assertEntry(actualOptional, Set.of(ROLE_ID));
    }

    @Test
    public void getRoles() {
        manager.addRole(idmAddRoleDto(ROLE_ID)).join();
        manager.addRole(idmAddRoleDto(ROLE_ID2)).join();
        manager.addRole(idmAddUserRoleDto(ROLE_ID3)).join();

        List<IdmRolesPageResponseDto.Role> roles = manager.getRoles().join();

        assertEquals(3, roles.size());
        var role1 = roles.stream().filter(role -> role.path.endsWith(ROLE_ID + "/")).findFirst().get();
        assertEquals(Integer.valueOf(GROUP), role1.group);
        assertEquals("/type/system/role/" + ROLE_ID + "/", role1.path);

        var role2 = roles.stream().filter(role -> role.path.endsWith(ROLE_ID2 + "/")).findFirst().get();
        assertEquals(Integer.valueOf(GROUP), role2.group);
        assertEquals("/type/system/role/" + ROLE_ID2 + "/", role2.path);

        var role3 = roles.stream().filter(role -> role.path.endsWith(ROLE_ID3 + "/")).findFirst().get();
        assertEquals(USER, role3.login);
        assertEquals("user", role3.subjectType);
        assertEquals("/type/system/role/" + ROLE_ID3 + "/", role3.path);
    }

    private void assertEntry(Optional<SystemAclEntry> actualOptional, Set<String> roles) {
        assertTrue(actualOptional.isPresent());
        SystemAclEntry entry = actualOptional.get();
        assertEquals(entry.getUid(), GROUP);
        assertEquals(entry.getType(), AclUidType.GROUP);
        assertEquals(entry.getRoles(), roles);
    }
}
