package ru.yandex.solomon.search;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.search.result.SearchResult;
import ru.yandex.solomon.search.roaring.BitmapIndex;


/**
 * Results (from solomon-dev-myt-00)
 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 *
 * Benchmark                                         Mode  Cnt      Score      Error  Units
 * IndexSearchBenchmark.generalOwn                  thrpt   10  17077.054 ±  506.826  ops/s
 * IndexSearchBenchmark.generalRoaring              thrpt   10   7723.633 ±  211.896  ops/s
 * IndexSearchBenchmark.manyUniqueMultiGlobOwn      thrpt   10     26.148 ±    1.895  ops/s
 * IndexSearchBenchmark.manyUniqueMultiGlobRoaring  thrpt   10   5714.975 ±  216.951  ops/s
 * IndexSearchBenchmark.manyUniqueOwn               thrpt   10     56.172 ±    2.050  ops/s
 * IndexSearchBenchmark.manyUniqueRoaring           thrpt   10  41734.749 ± 1708.664  ops/s
 *
 * @author Sergey Polovko
 */
@Fork(value = 1)
@Warmup(iterations = 20)
@Measurement(iterations = 10)
@BenchmarkMode(Mode.Throughput)
@OutputTimeUnit(TimeUnit.SECONDS)
public class IndexSearchBenchmark {

    private static final int METRIC_COUNT = 1_000_000;

    public static final class Case {
        private final Selectors selectors;
        private final BitmapIndex bitmapIndex;

        Case(List<Labels> metrics, int valuesCount) {
            this.selectors = StubMetrics.generateSelectors(metrics, valuesCount);
            this.bitmapIndex = BitmapIndex.build(metrics);
        }
    }

    @State(Scope.Benchmark)
    public static class BenchmarkState {
        /**
         * general random metrics
         */
        private final Case general = new Case(StubMetrics.generate(METRIC_COUNT), 1);

        /**
         * metrics with many unique values
         */
        private final Case manyUnique = new Case(StubMetrics.generateManyUniqueValues(METRIC_COUNT), 1);

        /**
         * metrics with many unique values with multi glob selectors
         */
        private final Case manyUniqueMultiGlob = new Case(StubMetrics.generateManyUniqueValues(METRIC_COUNT), 16);
    }

    @Benchmark
    public SearchResult generalRoaring(BenchmarkState state) {
        return state.general.bitmapIndex.search(state.general.selectors);
    }

    @Benchmark
    public SearchResult manyUniqueRoaring(BenchmarkState state) {
        return state.manyUnique.bitmapIndex.search(state.manyUnique.selectors);
    }

    @Benchmark
    public SearchResult manyUniqueMultiGlobRoaring(BenchmarkState state) {
        return state.manyUniqueMultiGlob.bitmapIndex.search(state.manyUniqueMultiGlob.selectors);
    }

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(IndexSearchBenchmark.class.getName())
            .detectJvmArgs()
            .jvmArgs("-Xmx2g", "-Xms2g")
            .build();

        new Runner(opt).run();
    }
}
