package ru.yandex.solomon.search;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.function.Function;
import java.util.stream.Collectors;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.search.result.SearchResult;
import ru.yandex.solomon.search.roaring.BitmapIndex;
import ru.yandex.solomon.util.timer.OncePerSecondRatePrinter;


/**
 * Run
 * ~~~
 *
 * OwnIndex
 *   dataset [10000000 elements, 266445760 bytes] built in 17763 ms
 *   50000 searches in 26385 ms
 *
 * BitmapIndex
 *   dataset [10000000 elements, 14913336 bytes] built in 6558 ms
 *   50000 searches in 73997 ms
 *
 * RoaringIndexSerialized
 *   dataset [10000000 elements, 14488964 bytes] built in 4140 ms
 *   50000 searches in 176082 ms
 *
 * @author Maksim Leonov
 */
public class SearchEnginePerf {

    private static final int METRIC_COUNT = 10_000_000;
    private static final int QUERY_COUNT = 50_000;

    private static final Random rnd = new Random(17);

    private static void doTest(Function<List<Labels>, SearchIndex> engine, List<Labels> metrics) {
        System.err.println("build index");
        long startTime = System.currentTimeMillis();
        SearchIndex search = engine.apply(metrics);

        System.out.printf("dataset [%d elements, %d bytes] built in %d ms\n",
            metrics.size(),
            search.getIndexSize(),
            System.currentTimeMillis() - startTime);

        System.err.println("prepare queries");
        List<Selectors> queries = metrics.stream()
            .map(Selectors::of)
            .limit(QUERY_COUNT)
            .collect(Collectors.toList());
        Collections.shuffle(queries, rnd);

        OncePerSecondRatePrinter opsp = new OncePerSecondRatePrinter(queries.size());
        System.err.println("start searching");

        long searchStartTime = System.currentTimeMillis();
        for (Selectors query : queries) {
            SearchResult result = search.search(query);
            if (result.size() != 1) {
                throw new RuntimeException("Wrong search result count: " + result.size());
            }
            opsp.add(1);
        }

        System.out.printf("%d searches in %d ms\n", queries.size(), System.currentTimeMillis() - searchStartTime);
    }

    private static void checkUnique(List<Labels> metrics) {
        HashSet<Labels> keys = new HashSet<>();
        for (Labels s : metrics) {
            if (!keys.add(s)) {
                throw new IllegalStateException("non unique key: " + s);
            }
        }
    }

    public static void main(String[] args) {
        List<Labels> metrics = StubMetrics.generate(METRIC_COUNT, rnd);
        // check unique
        checkUnique(metrics);

        System.out.println("BitmapIndex");
        SearchEnginePerf.doTest(BitmapIndex::build, metrics);
    }
}
