package ru.yandex.solomon.search.linear;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import it.unimi.dsi.fastutil.ints.IntArrayList;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.search.SearchIndex;
import ru.yandex.solomon.search.result.ArrayResult;
import ru.yandex.solomon.search.result.SearchResult;
import ru.yandex.solomon.util.labelStats.LabelStatsCollectors;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class LinearSearchIndex implements SearchIndex {

    private final List<Labels> targets;

    public LinearSearchIndex(List<Labels> targets) {
        this.targets = targets;
    }

    @Override
    public SearchResult search(Selectors selectors, int max) {
        IntArrayList r = new IntArrayList();
        for (int i = 0; i < targets.size() && i < max; i++) {
            Labels target = targets.get(i);
            if (selectors.match(target)) {
                r.add(i);
            }
        }
        return new ArrayResult(r.toIntArray());
    }

    @Override
    public ImmutableSet<String> labelNames() {
        HashSet<String> names = new HashSet<>();
        for (Labels target : targets) {
            target.forEach(label -> names.add(label.getKey()));
        }
        return ImmutableSet.copyOf(names);
    }

    @Override
    public LabelValuesStats labelStats(Set<String> requestedNames) {
        return targets.stream()
                .parallel()
                .collect(LabelStatsCollectors.toLabelValuesStats(requestedNames));
    }

    @Override
    public long getIndexSize() {
        return 0;
    }

    @Override
    public long getIndexCacheSize() {
        return 0;
    }

    public static LinearSearchIndex build(Collection<Labels> targets) {
        return new LinearSearchIndex(ImmutableList.copyOf(targets));
    }
}
