package ru.yandex.solomon.search.roaring;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.collect.ImmutableSet;
import io.netty.util.internal.RecyclableArrayList;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import org.apache.commons.lang3.mutable.MutableInt;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.search.SearchIndex;
import ru.yandex.solomon.search.result.EmptyResult;
import ru.yandex.solomon.search.result.RangeResult;
import ru.yandex.solomon.search.result.SearchResult;
import ru.yandex.solomon.util.labelStats.LabelStats;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class BitmapIndex implements SearchIndex {

    private final LabelKey[] labelKeys;
    private final ImmutableSet<String> labelKeysSet;
    private final long memorySize;
    private final int metricCount;

    private final Cache<Selector, Bitmap> cachedBitmapBySelector;
    private final Cache<Selectors, Bitmap> cacheBitmapBySelectors;
    private final AtomicLong cacheMemorySize = new AtomicLong();

    BitmapIndex(LabelKey[] labelKeys, int metricCount) {
        this.labelKeys = labelKeys;

        long memorySize = 0;
        ImmutableSet.Builder<String> b = ImmutableSet.builder();
        for (LabelKey labelKey : labelKeys) {
            b.add(labelKey.key);
            memorySize += labelKey.memorySizeIncludingSelf();
        }

        this.labelKeysSet = b.build();
        this.metricCount = metricCount;
        this.memorySize = memorySize;
        this.cachedBitmapBySelector = CacheBuilder.newBuilder()
                .maximumSize(100_000)
                .recordStats()
                .expireAfterAccess(10, TimeUnit.MINUTES)
                .removalListener((RemovalListener<Selector, Bitmap>) r -> cacheMemorySize.addAndGet(-r.getValue().memorySize()))
                .build();
        this.cacheBitmapBySelectors = CacheBuilder.newBuilder()
                .maximumSize(100_000)
                .recordStats()
                .expireAfterAccess(10, TimeUnit.MINUTES)
                .removalListener((RemovalListener<Selectors, Bitmap>) r -> cacheMemorySize.addAndGet(-r.getValue().memorySize()))
                .build();
    }

    public static BitmapIndex build(Collection<Labels> targets) {
        return build(targets.iterator());
    }

    public static BitmapIndex build(Iterator<Labels> targets) {
        Object2ObjectOpenHashMap<String, LabelKeyBuilder> keys = new Object2ObjectOpenHashMap<>(16);

        MutableInt idx = new MutableInt(0);
        while (targets.hasNext()) {
            Labels target = targets.next();
            target.forEach(l -> {
                LabelKeyBuilder key = keys.computeIfAbsent(l.getKey(), k -> new LabelKeyBuilder());
                key.add(idx.intValue(), l.getValue());
            });
            idx.increment();
        }

        int keyIdx = 0;
        LabelKey[] keysArr = new LabelKey[keys.size()];
        for (Map.Entry<String, LabelKeyBuilder> e : keys.entrySet()) {
            keysArr[keyIdx++] = e.getValue().build(e.getKey());
        }
        return new BitmapIndex(keysArr, idx.intValue());
    }

    @Override
    public SearchResult search(Selectors selectors, int max) {
        if (selectors.isEmpty()) {
            return allIds(max);
        }

        var cached = cacheBitmapBySelectors.getIfPresent(selectors);
        if (cached != null) {
            return Bitmaps.toSearchResult(cached, max);
        }

        List<Bitmap> inclusive = new ArrayList<>();
        List<Bitmap> exclusive = new ArrayList<>();

        for (Selector selector : selectors) {
            LabelKey labelKey = findLabelKey(selector.getKey());
            if (labelKey == null) {
                if (selector.match(null)) {
                    // is ABSENT or multi glob with ABSENT selector
                    continue;
                } else {
                    return EmptyResult.SELF;
                }
            }

            switch (selector.getType()) {
                case ANY:
                    inclusive.add(labelKey.bitmap);
                    break;

                case ABSENT:
                    exclusive.add(labelKey.bitmap);
                    break;

                default:
                    inclusive.add(cachedOrMatch(labelKey, selector));
                    break;
            }
        }

        if (inclusive.isEmpty() && exclusive.isEmpty()) {
            return allIds(max);
        }

        Bitmap bitmap = Bitmaps.combine(inclusive, exclusive, metricCount);
        if (selectors.size() >= 2 && bitmap.size() <= 10_000) {
            cacheMemorySize.addAndGet(bitmap.memorySize());
            cacheBitmapBySelectors.put(selectors, bitmap);
        }
        return Bitmaps.toSearchResult(bitmap, max);
    }

    @Nullable
    private LabelKey findLabelKey(String key) {
        for (LabelKey labelKey : labelKeys) {
            if (labelKey.key.equals(key)) {
                return labelKey;
            }
        }
        return null;
    }

    private Bitmap cachedOrMatch(LabelKey labelKey, Selector selector) {
        if (selector.isExact()) {
            return orMatch(labelKey, selector, false);
        }

        var cached = cachedBitmapBySelector.getIfPresent(selector);
        if (cached != null) {
            return cached;
        }

        var result = orMatch(labelKey, selector, true);
        cacheMemorySize.addAndGet(result.memorySize());
        cachedBitmapBySelector.put(selector, result);
        return result;
    }

    private Bitmap orMatch(LabelKey labelKey, Selector selector, boolean runOptimize) {
        RecyclableArrayList matched = RecyclableArrayList.newInstance();
        try {
            selector.forEachMatchedKey(labelKey.values, matched::add);

            // if selector has at least one ABSENT in multi glob
            // then add all metrics which has no current label key
            if (selector.match(null)) {
                matched.add(Bitmaps.invert(labelKey.bitmap, metricCount));
            }

            switch (matched.size()) {
                case 0:
                    return EmptyBitmap.INSTANCE;
                case 1:
                    return (Bitmap) matched.get(0);
                default:
                    //noinspection unchecked
                    return Bitmaps.or((List) matched, runOptimize);
            }
        } finally {
            matched.recycle();
        }
    }

    private SearchResult allIds(int max) {
        int len = Math.min(Math.max(0, max), metricCount);
        if (len == 0) {
            return EmptyResult.SELF;
        }
        return new RangeResult(0, len);
    }

    @Override
    public ImmutableSet<String> labelNames() {
        return labelKeysSet;
    }

    @Override
    public LabelValuesStats labelStats(Set<String> requestedNames) {
        int expectedSize = requestedNames.isEmpty() ? labelKeys.length : requestedNames.size();

        Map<String, LabelStats> statsByLabelKey = new HashMap<>(expectedSize);
        if (requestedNames.isEmpty()) {
            for (LabelKey labelKey : labelKeys) {
                statsByLabelKey.put(labelKey.key, toLabelStats(labelKey));
            }
        } else {
            for (LabelKey labelKey : labelKeys) {
                if (requestedNames.contains(labelKey.key)) {
                    statsByLabelKey.put(labelKey.key, toLabelStats(labelKey));
                }
            }
        }

        return new LabelValuesStats(statsByLabelKey, metricCount);
    }

    private static LabelStats toLabelStats(LabelKey labelKey) {
        return new LabelStats(labelKey.values.keySet(), labelKey.metricCount, false);
    }

    @Override
    public long getIndexSize() {
        return memorySize;
    }

    @Override
    public long getIndexCacheSize() {
        return cacheMemorySize.get();
    }

    /**
     * LABEL KEY
     */
    private static final class LabelKey implements MemMeasurable {
        private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(LabelKey.class);

        private final String key;
        private final Bitmap bitmap;
        private final Object2ObjectOpenHashMap<String, Bitmap> values;
        private final int metricCount;

        LabelKey(String key, Bitmap bitmap, Object2ObjectOpenHashMap<String, Bitmap> values) {
            this.key = key;
            this.bitmap = bitmap;
            this.values = values;
            this.metricCount = bitmap.size();
        }

        @Override
        public long memorySizeIncludingSelf() {
            long valuesSize = 0;
            for (Map.Entry<String, Bitmap> e : values.entrySet()) {
                valuesSize += MemoryCounter.stringSize(e.getKey());
                valuesSize += e.getValue().memorySize();
            }
            return SELF_SIZE
                + MemoryCounter.stringSize(key)
                + bitmap.memorySize()
                + MemoryCounter.object2ObjectOpenHashMapSize(values)
                + valuesSize;
        }
    }

    /**
     * LABEL KEY BUILDER
     */
    private static final class LabelKeyBuilder {
        private final RoaringBitmap keyBitmap = new RoaringBitmap();
        private final Object2ObjectOpenHashMap<String, RoaringBitmap> valueBitmaps = new Object2ObjectOpenHashMap<>();

        public void add(int metricId, String value) {
            RoaringBitmap valueBitmap = valueBitmaps.get(value);
            if (valueBitmap == null) {
                valueBitmaps.put(value, valueBitmap = new RoaringBitmap());
            }
            keyBitmap.add(metricId);
            valueBitmap.add(metricId);
        }

        public LabelKey build(String key) {
            var valueBitmaps = new Object2ObjectOpenHashMap<String, Bitmap>(this.valueBitmaps.size());
            for (Map.Entry<String, RoaringBitmap> e : this.valueBitmaps.entrySet()) {
                valueBitmaps.put(e.getKey(), Bitmaps.optimize(e.getValue()));
            }

            Bitmap keyBitmap = Bitmaps.optimize(this.keyBitmap);
            return new LabelKey(key, keyBitmap, valueBitmaps);
        }
    }
}
