package ru.yandex.solomon.search.roaring;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.roaringbitmap.FastAggregation;

import ru.yandex.solomon.search.result.EmptyResult;
import ru.yandex.solomon.search.result.SearchResult;


/**
 * @author Sergey Polovko
 */
final class Bitmaps {
    private Bitmaps() {}

    static SearchResult toSearchResult(Bitmap bitmap, int limit) {
        if (limit <= 0 || bitmap.size() == 0) {
            return EmptyResult.SELF;
        }
        return new BitmapResult(bitmap, limit);
    }

    static Bitmap optimize(RoaringBitmap bitmap) {
        final int size = bitmap.size();
        if (size == 0) {
            return EmptyBitmap.INSTANCE;
        } else if (size == 1) {
            return new SingleBitmap(bitmap.first());
        } else if (size <= 10) {
            return new ArrayBitmap(bitmap.toArray());
        } else {
            bitmap.runOptimize();
            return bitmap;
        }
    }

    /**
     * See {@link FastAggregation#or(org.roaringbitmap.RoaringBitmap...)}
     */
    static Bitmap or(Collection<Bitmap> bitmaps) {
        return or(bitmaps, false);
    }

    static Bitmap or(Collection<Bitmap> bitmaps, boolean runOptimize) {
        if (bitmaps.isEmpty()) {
            return EmptyBitmap.INSTANCE;
        }

        // It is way more faster to add all RoaringBitmaps lazily and
        // repair resulting bitmap just once. Unfortunately not yet
        // repaired bitmap cannot be used except for lazy operations.

        RoaringBitmap answer = new RoaringBitmap();
        for (Bitmap bitmap : bitmaps) {
            if (bitmap instanceof RoaringBitmap) {
                answer.lazyOr((RoaringBitmap) bitmap);
            }
        }
        answer.repairAfterLazyOr();
        for (Bitmap bitmap : bitmaps) {
            if (!(bitmap instanceof RoaringBitmap)) {
                bitmap.addTo(answer);
            }
        }
        if (runOptimize) {
            answer.runOptimize();
        }
        return answer;
    }

    /**
     * See {@link FastAggregation#and(org.roaringbitmap.RoaringBitmap...)}
     */
    private static RoaringBitmap and(Collection<Bitmap> bitmaps) {
        if (bitmaps.isEmpty()) {
            return new RoaringBitmap();
        }
        Iterator<Bitmap> it = bitmaps.iterator();
        RoaringBitmap answer = RoaringBitmap.copyOf(it.next());
        while (it.hasNext()) {
            answer.and(it.next());
        }
        return answer;
    }

    static Bitmap combine(List<Bitmap> inclusive, List<Bitmap> exclusive, int metricCount) {
        if (exclusive.isEmpty()) {
            // (1) only inclusions
            return (inclusive.size() > 1)
                ? and(inclusive)
                : inclusive.get(0);
        }

        if (inclusive.isEmpty()) {
            // (2) only exclusions
            return invert(or(exclusive), metricCount);
        }

        // (3) intersection
        RoaringBitmap bitmap = (inclusive.size() > 1)
            ? and(inclusive)
            : RoaringBitmap.copyOf(inclusive.get(0));
        for (Bitmap exc : exclusive) {
            exc.removeFrom(bitmap);
        }
        return bitmap;
    }

    static Bitmap invert(Bitmap bitmap, int maxSize) {
        RoaringBitmap all = new RoaringBitmap();
        all.add(0L, (long) maxSize);
        bitmap.removeFrom(all);
        return all;
    }
}
