package ru.yandex.solomon.search;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableList;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.misc.random.Random2;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.search.linear.LinearSearchIndex;


/**
 * @author Stepan Koltsov
 */
public abstract class SearchIndexRandomTest extends SearchIndexAnyTest {

    private Random random = new Random(17);

    private String randomName() {
        return new Random2(random)
            .randomElement(new String[] { "aaa", "ab", "bb", "c", "cc" });
    }

    private String randomValue() {
        return new Random2(random)
            .randomElement(new String[] { "aa", "ab", "ac", "ba", "bbc", "cc", "ccc" });
    }

    private Label randomLabel() {
        return Labels.allocator.alloc(randomName(), randomValue());
    }

    private String randomQueryValue() {
        if (random.nextInt(5) == 0) {
            return "-";
        }

        int alterations = 1 + random.nextInt(3);

        ArrayList<String> globs = new ArrayList<>();
        for (int i = 0; i < alterations; ++i) {
            globs.add(new Random2(random)
                .randomElement(new String[] { "-", "a*", "*a", "*", "b*c", "*c*" }));
        }

        return String.join("|", globs);
    }

    private Selector randomSelector() {
        String name = randomName();
        String value = randomQueryValue();
        boolean isNegative = random.nextInt(4) == 0;

        return isNegative ? Selector.notGlob(name, value) : Selector.notGlob(name, value);
    }

    private Labels randomLabelList() {
        int length = random.nextInt(4);
        LabelsBuilder builder = Labels.builder(length);

        for (int i = 0; i < length; ) {
            Label label = randomLabel();
            if (builder.hasKey(label.getKey())) {
                continue;
            }

            builder.add(label);
            ++i;
        }

        return builder.build();
    }

    private Selectors randomQuery() {
        HashSet<String> unique = new HashSet<>();

        ArrayList<Selector> selectors = new ArrayList<>();

        int count = random.nextInt(5);
        for (int i = 0; i < count; ++i) {
            Selector selector = randomSelector();
            if (!unique.add(selector.getKey())) {
                continue;
            }

            selectors.add(selector);
        }

        return Selectors.of(selectors);
    }

    private void randomIteration() {
        HashSet<Labels> unique = new HashSet<>();
        for (int i = 0; i < 20; ++i) {
            unique.add(randomLabelList());
        }

        ImmutableList<Labels> docs = ImmutableList.copyOf(unique);

        SearchIndex index = build(unique);
        LinearSearchIndex linear = LinearSearchIndex.build(docs);

        for (int i = 0; i < 50; ++i) {
            Selectors query = randomQuery();
            int[] expected = linear.search(query).toArray();
            int[] found = index.search(query).toArray();

            System.out.println("selectors: " + query);
            System.out.println("expected:  " + getLabelsByRefs(docs, expected));
            System.out.println("found:     " + getLabelsByRefs(docs, found));
            System.out.println("docs:     " + docs);

            Assert.assertArrayEquals(expected, found);
        }
    }

    private static List<Labels> getLabelsByRefs(List<Labels> docs, int[] expected) {
        return Arrays.stream(expected)
            .mapToObj(docs::get)
            .collect(Collectors.toList());
    }

    @Test
    public void random() {
        for (int i = 0; i < 200; ++i) {
            randomIteration();
        }
    }
}
