package ru.yandex.solomon.search;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;

import it.unimi.dsi.fastutil.ints.IntArrayList;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaIgnore;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.util.labelStats.LabelStats;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;


/**
 * @author Stepan Koltsov
 */
public abstract class SearchIndexTest extends SearchIndexAnyTest {

    private SearchIndex search;
    private List<Labels> labelLists;

    protected void makeSearch(Labels... labelSets) {
        makeSearch(Arrays.asList(labelSets));
    }

    protected void makeSearch(List<Labels> labelLists) {
        this.labelLists = labelLists;
        search = build(labelLists);
    }

    private int[] search(String selectors) {
        return search.search(Selectors.parse(selectors)).toArray();
    }

    private int[] search(Selectors selectors) {
        return search.search(selectors).toArray();
    }

    private int[] searchWithSelectors(String selectors) {
        Selectors parsedSelectors = Selectors.parse(selectors);

        IntArrayList r = new IntArrayList();
        for (int i = 0; i < labelLists.size(); ++i) {
            if (parsedSelectors.match(labelLists.get(i))) {
                r.add(i);
            }
        }
        return r.toIntArray();
    }

    private int[] searchWithSelectors(Selectors selectors) {
        IntArrayList r = new IntArrayList();
        for (int i = 0; i < labelLists.size(); ++i) {
            if (selectors.match(labelLists.get(i))) {
                r.add(i);
            }
        }
        return r.toIntArray();
    }

    private void searchAndTest(String query, int... expected) {
        // testing search
        int[] searchResult = search(query);
        Assert.assertArrayEquals(expected, searchResult);

        // testing selectors
        int[] filterResult = searchWithSelectors(query);
        Assert.assertArrayEquals(expected, filterResult);
    }

    private void searchAndTest(Selectors query, int... expected) {
        // testing search
        int[] searchResult = search(query);
        Assert.assertArrayEquals(expected, searchResult);

        // testing selectors
        int[] filterResult = searchWithSelectors(query);
        Assert.assertArrayEquals(expected, filterResult);
    }

    @Test
    public void emptySelector() {
        makeSearch(
            Labels.of("host", "aaa", "sensor", "xxx"),
            Labels.of("host", "bbb", "sensor", "xxx"),
            Labels.of("host", "aaa", "sensor", "yyy"),
            Labels.of("host", "bbb", "sensor", "yyy"));
        searchAndTest("", 0, 1, 2, 3);
    }

    @Test
    public void simple() {
        makeSearch(
            Labels.of("host", "aaa", "sensor", "xxx"),
            Labels.of("host", "bbb", "sensor", "xxx"),
            Labels.of("host", "aaa", "sensor", "yyy"),
            Labels.of("host", "bbb", "sensor", "yyy"));
        searchAndTest("host=aaa, sensor=xxx", 0);
        searchAndTest("host=bbb", 1, 3);
    }

    @Test
    public void ne() {
        makeSearch(
            Labels.of("host", "aaa", "sensor", "xxx"),
            Labels.of("host", "bbb", "sensor", "xxx"),
            Labels.of("host", "aaa", "sensor", "yyy"),
            Labels.of("host", "bbb", "sensor", "yyy"));
        searchAndTest("host!=bbb", 0, 2);
        searchAndTest("host=aaa, sensor!=xxx", 2);
    }

    @Test
    public void someLabelsMissing() {
        makeSearch(
                Labels.of("host", "aaa", "sensor", "xxx"),
                Labels.of("host", "bbb", "sensor", "xxx"),
                Labels.of("host", "aaa", "sensor", "yyy", "user", "nnn"),
                Labels.of("host", "aaa", "sensor", "yyy", "user", "mmm"));
        searchAndTest("user=nnn", 2);
        searchAndTest("user!=nnn", 3);
        searchAndTest("user=*", 2, 3);
        searchAndTest("user=-", 0, 1);
    }

    @Test
    public void excludeNonexistentLabel() {
        makeSearch(
            Labels.of("aa", "bb", "cc", "dd"),
            Labels.of("aa", "cc", "bb", "dd"));

        searchAndTest("xx=-", 0, 1);
        searchAndTest("xx=value|-", 0, 1);
        searchAndTest("xx=*|-", 0, 1);
    }

    @Test
    public void nonexistentLabelWithNegative() {
        makeSearch(
            Labels.of("aa", "bb", "cc", "dd"),
            Labels.of("aa", "cc", "bb", "dd"));

        searchAndTest("xx!=value");
        searchAndTest("xx!=value, aa=bb");
    }

    @Test
    public void existentAndExcludeNonexistentLabel() {
        makeSearch(
            Labels.of("aa", "bb", "cc", "dd"),
            Labels.of("aa", "cc", "bb", "dd"));

        searchAndTest("aa=bb, xx=-", 0);
    }

    @Test
    public void multiGlobWithAbsent() {
        makeSearch(
            Labels.of("aa", "bb", "cc", "dd"),
            Labels.of("aa", "cc", "bb", "dd"),
            Labels.of("bb", "dd"),
            Labels.of("dd", "ee"));

        searchAndTest("aa=bb|-", 0, 2, 3);
        searchAndTest("aa=b*|-", 0, 2, 3);
        searchAndTest("aa=*c|-", 1, 2, 3);
        searchAndTest("aa=*|-", 0, 1, 2, 3);
    }

    @Test
    public void substring() {
        makeSearch(
            Labels.of("host", "aaa001", "sensor", "xxxnnn"),
            Labels.of("host", "aaa002", "sensor", "yyymmm"),
            Labels.of("host", "bbb001", "sensor", "xxxmmm"),
            Labels.of("host", "bbb002", "sensor", "yyynnn"));
        searchAndTest("host=aaa*", 0, 1);
        searchAndTest("sensor=*nnn", 0, 3);
    }

    @Test
    public void alternate() {
        makeSearch(
                Labels.of("host", "aaa001", "sensor", "xxxnnn"),
                Labels.of("host", "aaa002", "sensor", "yyymmm"),
                Labels.of("host", "bbb001", "sensor", "xxxmmm"),
                Labels.of("host", "bbb002", "sensor", "yyynnn"));
        searchAndTest("host=aaa001|aaa002", 0, 1);
        searchAndTest("host=a*1|b*2", 0, 3);
        searchAndTest("sensor!=xxxmmm|y*n", 0, 1);
    }

    @Test
    public void full() {
        ArrayList<Labels> queries = new ArrayList<>();
        for (int host = 0; host < 20; ++host) {
            for (int metric = 0; metric < 20; ++metric) {
                for (int smth = 0; smth < 20; ++smth) {
                    Labels labels = Labels.of(
                        "host", Integer.toString(host),
                        "sensor", Integer.toString(metric),
                        "smth", Integer.toString(smth));
                    queries.add(labels);
                }
            }
        }

        makeSearch(queries);

        Random random = new Random(17);
        for (int i = 0; i < 100; ++i) {
            int doc = random.nextInt(queries.size());
            Labels query = queries.get(doc);
            searchAndTest(Selectors.of(query), doc);
        }
    }

    @Test
    public void limit() {
        makeSearch(
            Labels.of("a", "b"),
            Labels.of("a", "c"),
            Labels.of("a", "d"));

        int[] searchResult = search.search(Selectors.parse("a=*"), 1).toArray();
        Assert.assertEquals(1, searchResult.length);
    }

    @Test
    public void nonExistentLabelName() {
        makeSearch(
            Labels.of("a", "b", "c", "d"),
            Labels.of("a", "c", "d", "f"));

        searchAndTest("x=y");
    }

    @Test
    public void nonExistentLabelValue() {
        makeSearch(
            Labels.of("a", "b", "c", "d"),
            Labels.of("a", "x", "c", "y"));

        searchAndTest("a=nonex");
    }

    @Test
    public void emptyResult() {
        makeSearch(
            Labels.of("a", "b", "c", "d"),
            Labels.of("a", "y", "c", "y"));

        searchAndTest("a=y, c=d");
    }

    @Test
    public void labelInAlmostAllDocs() {
        makeSearch(
            Labels.of("a", "10", "b", "x"),
            Labels.of("a", "10", "b", "y"),
            Labels.of("a", "10", "b", "z"),
            Labels.of("a", "20", "b", "w"));

        searchAndTest("a=10", 0, 1, 2);
    }

    @Test
    public void labelNames() {
        makeSearch(
            Labels.of("host", "aaa", "sensor", "xxx"),
            Labels.of("host", "bbb", "sensor", "xxx"),
            Labels.of("host", "aaa", "sensor", "yyy", "user", "nnn"),
            Labels.of("host", "aaa", "sensor", "yyy", "user", "mmm"));

        Set<String> actual = search.labelNames();
        Set<String> expected = new HashSet<>(Arrays.asList("host", "sensor", "user"));

        Assert.assertEquals(expected, actual);
    }

    @YaIgnore
    @Test
    public void labelStats() {
        makeSearch(
            Labels.of("host", "aaa", "sensor", "xxx"),
            Labels.of("host", "bbb", "sensor", "xxx"),
            Labels.of("host", "aaa", "sensor", "yyy", "user", "nnn"),
            Labels.of("host", "aaa", "sensor", "yyy", "user", "mmm"));

        LabelValuesStats actualStats = search.labelStats(Collections.emptySet());

        HashMap<String, LabelStats> statsByLabelKey = new HashMap<>();
        statsByLabelKey.put("host", newLabelStats(4, "aaa", "bbb"));
        statsByLabelKey.put("sensor", newLabelStats(4, "xxx", "yyy"));
        statsByLabelKey.put("user", newLabelStats(2, "nnn", "mmm"));

        LabelValuesStats expectedStats = new LabelValuesStats(statsByLabelKey, 4);

        Assert.assertEquals(expectedStats, actualStats);
    }

    @YaIgnore
    @Test
    public void labelStatsWithRestriction() {
        makeSearch(
            Labels.of("host", "aaa", "sensor", "xxx"),
            Labels.of("host", "bbb", "sensor", "xxx"),
            Labels.of("host", "aaa", "sensor", "yyy", "user", "ppp"),
            Labels.of("host", "aaa", "sensor", "yyy", "user", "qqq"));

        LabelValuesStats actualStats = search.labelStats(Collections.singleton("user"));

        HashMap<String, LabelStats> statsByLabelKey = new HashMap<>();
        statsByLabelKey.put("user", newLabelStats(2, "ppp", "qqq"));

        LabelValuesStats expectedStats = new LabelValuesStats(statsByLabelKey, 4);

        Assert.assertEquals(expectedStats, actualStats);
    }

    private static LabelStats newLabelStats(int metricsCount, String... values) {
        return new LabelStats(new HashSet<>(Arrays.asList(values)), metricsCount, false);
    }
}
