package ru.yandex.solomon.secrets;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.protobuf.TextFormat;

import ru.yandex.solomon.config.protobuf.Secret;
import ru.yandex.solomon.config.protobuf.SecretsConfig;
import ru.yandex.solomon.staffOnly.annotations.HideFromManagerUi;

/**
 * @author Sergey Polovko
 */
@HideFromManagerUi
final class SecretProviderImpl implements SecretProvider {

    public static final String YAV_SCHEME = "yav://";
    private final ImmutableMap<String, String> secrets;

    SecretProviderImpl(SecretsConfig config) {
        var secretsBuilder = ImmutableMap.<String, String>builder();
        for (Secret secret : config.getSecretsList()) {
            secretsBuilder.put(secret.getKey(), secret.getValue());
        }
        secrets = secretsBuilder.build();
    }

    public static SecretProvider fromFile(Path path) {
        Preconditions.checkArgument(Files.exists(path), "file %s does not exist", path);
        try {
            return fromString(Files.readString(path, Charsets.UTF_8));
        } catch (IOException e) {
            throw new UncheckedIOException("cannot read file " + path, e);
        }
    }

    public static SecretProvider fromString(String content) {
        return new SecretProviderImpl(parseConfig(content));
    }

    @Override
    public Optional<String> getSecret(String key) {
        if (key.startsWith(YAV_SCHEME)) {
            key = key.substring(YAV_SCHEME.length());
        }
        return Optional.ofNullable(secrets.get(key));
    }

    private static SecretsConfig parseConfig(String content) {
        try {
            SecretsConfig.Builder configBuilder = SecretsConfig.newBuilder();
            TextFormat.merge(content, configBuilder);
            return configBuilder.build();
        } catch (TextFormat.ParseException e) {
            throw new RuntimeException("cannot parse " + SecretsConfig.getDescriptor().getFullName(), e);
        }
    }
}
