package ru.yandex.solomon.selfmon.counters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;


/**
 * @author Sergey Polovko
 */
public class AsyncMetrics {

    private final Rate started;
    private final Rate completedOk;
    private final Rate completedError;
    private final LazyGaugeInt64 inFlight;
    private final Histogram responseTimeMillis;

    public AsyncMetrics(MetricRegistry registry, String namePrefix) {
        this(registry, namePrefix, Labels.empty());
    }

    public AsyncMetrics(MetricRegistry registry, String namePrefix, Labels labels) {
        if (!namePrefix.endsWith(".")) {
            namePrefix = namePrefix + '.';
        }
        started = registry.rate(namePrefix + "started", labels);
        completedOk = registry.rate(namePrefix + "completedOk", labels);
        completedError = registry.rate(namePrefix + "completedError", labels);
        inFlight = registry.lazyGaugeInt64(namePrefix + "inFlight", labels, this::getInFlight);
        responseTimeMillis = registry.histogramRate(namePrefix + "elapsedTimeMs", labels, Histograms.exponential(15, 2, 16));
    }

    public long getStarted() {
        return started.get();
    }

    public long getCompletedOk() {
        return completedOk.get();
    }

    public long getCompletedError() {
        return completedError.get();
    }

    public long getInFlight() {
        return getStarted() - getCompletedOk() - getCompletedError();
    }

    public void callStarted() {
        callStarted(1);
    }

    public void callStarted(long count) {
        started.add(count);
    }

    public void callCompletedOk(long elapsedMillis) {
        callCompletedOk(elapsedMillis, 1);
    }

    public void callCompletedOk(long elapsedMillis, long count) {
        completedOk.add(count);
        responseTimeMillis.record(elapsedMillis, count);
    }

    public void callCompletedError(long elapsedMillis) {
        callCompletedError(elapsedMillis, 1);
    }

    public void callCompletedError(long elapsedMillis, long count) {
        completedError.add(count);
        responseTimeMillis.record(elapsedMillis, count);
    }

    public void forFuture(CompletableFuture<?> future) {
        forFuture(future, 1);
    }

    public void forFuture(CompletableFuture<?> future, int count) {
        long startTimeNanos = System.nanoTime();
        callStarted(count);
        future.whenComplete((r, t) -> {
            long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos);
            if (t != null) {
                callCompletedError(elapsedMillis, count);
            } else {
                callCompletedOk(elapsedMillis, count);
            }
        });
    }

    public <T> CompletableFuture<T> wrapFuture(CompletableFuture<T> future) {
        return wrapFuture(future, 1);
    }

    public <T> CompletableFuture<T> wrapFuture(CompletableFuture<T> future, int count) {
        long startTimeNanos = System.nanoTime();
        callStarted(count);
        return future.whenComplete((r, t) -> {
            long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos);
            if (t != null) {
                callCompletedError(elapsedMillis, count);
            } else {
                callCompletedOk(elapsedMillis, count);
            }
        });
    }

    public <T> CompletableFuture<T> wrapFuture(Supplier<CompletableFuture<T>> supplier) {
        return wrapFuture(supplier, 1);
    }

    public <T> CompletableFuture<T> wrapFuture(Supplier<CompletableFuture<T>> supplier, int count) {
        long startTimeNanos = System.nanoTime();
        callStarted(count);
        try {
            return supplier.get().whenComplete((r, t) -> {
                long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos);
                if (t != null) {
                    callCompletedError(elapsedMillis, count);
                } else {
                    callCompletedOk(elapsedMillis, count);
                }
            });
        } catch (Throwable e) {
            long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos);
            callCompletedError(elapsedMillis, count);
            return CompletableFuture.failedFuture(e);
        }
    }

    public void combine(AsyncMetrics rhs) {
        this.started.combine(rhs.started);
        this.completedOk.combine(rhs.completedOk);
        this.completedError.combine(rhs.completedError);
        this.responseTimeMillis.combine(rhs.responseTimeMillis);
    }
}
