package ru.yandex.solomon.selfmon.failsafe;

/**
 * One of the use pattern:
 * <pre>
 *     if (!circuitBreaker.attemptExecution()) {
 *         throw new IllegalStateException("CircuitBreaker open");
 *     }
 *     try {
 *          // call external system
 *          circuitBreaker.markSuccess();
 *     } catch (Throwable e) {
 *          circuitBreaker.markFailure();
 *          throw e;
 *     }
 * </pre>
 *
 * @author Vladimir Gordiychuk
 * @see <a href="https://martinfowler.com/bliki/CircuitBreaker.html">CircuitBreaker</a>
 */
public interface CircuitBreaker {
    /**
     * Report about success request
     */
    void markSuccess();

    /**
     * Report about failed request and move to {@link Status#OPEN} if failure threshold reached
     */
    void markFailure();

    /**
     * Should be invoke before logic wrapped into CircuitBreaker and if result {@code false}
     * interrupt executing logic, because CircuitBreaker open.
     *
     * @return {@code true} request can be executed
     */
    boolean attemptExecution();

    /**
     * Human readable summary about circuit breaker state
     */
    String getSummary();

    /**
     * Current state of circuit breaker. This method not changed inner state and if status
     * equal to {@link Status#OPEN} it doesn't means that next call
     * {@link CircuitBreaker#attemptExecution()} return {@code false}
     */
    Status getStatus();

    enum Status {
        CLOSED, OPEN, HALF_OPEN
    }
}
