package ru.yandex.solomon.selfmon.failsafe;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

import ru.yandex.monlib.metrics.meter.ExpMovingAverage;

/**
 * Open when percent of failures on 1 minute window reach threshold. As a window function uses
 * exponentially-weighted moving average. Percent of failures updates every 5 second.
 *
 * @see ExpMovingAverage
 * @author Vladimir Gordiychuk
 */
public final class ExpMovingAverageCircuitBreaker implements CircuitBreaker {
    private static final long TICK_INTERVAL = ExpMovingAverage.oneMinute().getTickIntervalNanos();

    private final ExpMovingAverage fail = ExpMovingAverage.oneMinute();
    private final ExpMovingAverage success = ExpMovingAverage.oneMinute();

    private final double failureThresholdPercent;
    private final long resetTimeoutNanos;

    private final AtomicLong lastTick;
    private final AtomicReference<Status> status = new AtomicReference<>(Status.CLOSED);
    private final AtomicLong circuitOpenTime = new AtomicLong(-1);
    private final Clock clock;

    public ExpMovingAverageCircuitBreaker(double failureThresholdPercent, long resetTimeoutMillis) {
        this(SystemClock.INSTANCE, failureThresholdPercent, resetTimeoutMillis);
    }

    public ExpMovingAverageCircuitBreaker(Clock clock, double failureThresholdPercent, long resetTimeoutMillis) {
        this.clock = clock;
        this.lastTick = new AtomicLong(clock.nanoTime());
        this.failureThresholdPercent = failureThresholdPercent;
        this.resetTimeoutNanos = TimeUnit.MILLISECONDS.toNanos(resetTimeoutMillis);
    }

    @Override
    public void markSuccess() {
        tickIfNecessary();
        success.inc();

        if (status.compareAndSet(Status.HALF_OPEN, Status.CLOSED)) {
            resetMetrics();
            circuitOpenTime.set(-1L);
        }
    }

    @Override
    public void markFailure() {
        tickIfNecessary();
        fail.inc();

        if (status.compareAndSet(Status.HALF_OPEN, Status.OPEN)) {
            circuitOpenTime.set(clock.nanoTime());
            return;
        }

        double failRate = fail.getRate(TimeUnit.SECONDS);
        if (Double.compare(failRate, 0d) == 0) {
            return;
        }

        if (Double.compare(getFailurePercent(), failureThresholdPercent) < 0) {
            return;
        }

        // our failure rate is too high, we need to set the state to OPEN
        if (status.compareAndSet(Status.CLOSED, Status.OPEN)) {
            circuitOpenTime.set(clock.nanoTime());
        }
    }

    @Override
    public boolean attemptExecution() {
        long openTime = circuitOpenTime.get();
        if (openTime == -1) {
            return true;
        }

        if (status.get() == Status.HALF_OPEN) {
            return false;
        }

        if ((clock.nanoTime() - openTime) > resetTimeoutNanos) {
            return status.compareAndSet(Status.OPEN, Status.HALF_OPEN);
        }

        return false;
    }

    @Override
    public String getSummary() {
        return getFailurePercent() + " errors/second";
    }

    private void tickIfNecessary() {
        final long oldTick = lastTick.get();
        final long newTick = clock.nanoTime();
        final long elapsed = newTick - oldTick;
        if (elapsed >= TICK_INTERVAL) {
            final long newIntervalStartTick = newTick - elapsed % TICK_INTERVAL;
            if (lastTick.compareAndSet(oldTick, newIntervalStartTick)) {
                final long requiredTicks = elapsed / TICK_INTERVAL;
                for (long i = 0; i < requiredTicks; i++) {
                    fail.tick();
                    success.tick();
                }
            }
        }
    }

    @Override
    public Status getStatus() {
        return status.get();
    }

    /**
     * Percent of errors per second by current circuit breaker, useful for monitoring expose
     */
    public double getFailurePercent() {
        double failRate = fail.getRate(TimeUnit.SECONDS);
        if (Double.compare(failRate, 0d) == 0) {
            return 0d;
        }

        double successRate = success.getRate(TimeUnit.SECONDS);
        return failRate / (successRate + failRate);
    }

    private void resetMetrics() {
        fail.reset();
        success.reset();
        lastTick.set(clock.nanoTime());
    }

    @Override
    public String toString() {
        return "ExpMovingAverageCircuitBreaker{" + getSummary() + "}";
    }
}
