package ru.yandex.solomon.selfmon.failsafe;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Open when sequential count failures reach provided threshold. This CircuitBreaker more sensitive
 * to failures, because for example when 90% of request success and only 3 sequential request failed
 * CircuitBreaker will be open.
 *
 * @author Vladimir Gordiychuk
 */
public final class SimpleCircuitBreaker implements CircuitBreaker {

    private final AtomicReference<Status> status = new AtomicReference<>(Status.CLOSED);
    private final AtomicLong failCount = new AtomicLong();
    private final AtomicLong latestFailTime = new AtomicLong();

    private final long failureThreshold;
    private final long resetTimeoutNano;
    private final Clock clock;

    public SimpleCircuitBreaker(long failureThreshold, long resetTimeoutMillis) {
        this(SystemClock.INSTANCE, failureThreshold, resetTimeoutMillis);
    }

    public SimpleCircuitBreaker(Clock clock, long failureThreshold, long resetTimeoutMillis) {
        this.clock = clock;
        this.failureThreshold = failureThreshold;
        this.resetTimeoutNano = TimeUnit.MILLISECONDS.toNanos(resetTimeoutMillis);
    }

    @Override
    public void markSuccess() {
        failCount.set(0);
        latestFailTime.set(-1);
        status.compareAndSet(Status.HALF_OPEN, Status.CLOSED);
    }

    @Override
    public void markFailure() {
        long fail = failCount.incrementAndGet();
        latestFailTime.set(clock.nanoTime());

        if (!status.compareAndSet(Status.HALF_OPEN, Status.OPEN)) {
            if (fail >= failureThreshold) {
                status.set(Status.OPEN);
            }
        }
    }

    @Override
    public boolean attemptExecution() {
        if (status.get() == Status.CLOSED) {
            return true;
        }

        if (status.get() == Status.HALF_OPEN) {
            return false;
        }

        if (clock.nanoTime() - latestFailTime.get() > resetTimeoutNano) {
            return status.compareAndSet(Status.OPEN, Status.HALF_OPEN);
        }

        return false;
    }

    @Override
    public String getSummary() {
        return "Fails " + failCount.get() + " for the latest seconds "
                + TimeUnit.NANOSECONDS.toSeconds(latestFailTime.get());
    }

    @Override
    public Status getStatus() {
        return status.get();
    }
}
