package ru.yandex.solomon.selfmon.failsafe;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class TokenBucketRateLimiter implements RateLimiter {
    private static final long NANOTOKENS_IN_TOKEN = 1_000_000_000;
    private static final double TOKENS_IN_NANOTOKENS = 1d / NANOTOKENS_IN_TOKEN;

    private final Clock clock;
    private final long capacityNanotokens;
    private final double refillNanotokensPerMillis;

    private final AtomicLong availableNanotokens = new AtomicLong(0);
    private final AtomicLong lastRefill = new AtomicLong(0);

    public TokenBucketRateLimiter(Clock clock, int bucketSize, Duration window) {
        this.clock = clock;
        capacityNanotokens = bucketSize * NANOTOKENS_IN_TOKEN;
        refillNanotokensPerMillis = (double) capacityNanotokens / window.toMillis();

        availableNanotokens.set(capacityNanotokens);
        lastRefill.set(clock.millis());
    }

    public boolean acquire(int tokens) {
        final long nanotokens = tokens * NANOTOKENS_IN_TOKEN;
        refill();

        long availableNanotokens;
        do {
            availableNanotokens = this.availableNanotokens.get();
            if (availableNanotokens < nanotokens) {
                return false;
            }
        } while (!this.availableNanotokens.compareAndSet(availableNanotokens, availableNanotokens - nanotokens));

        return true;
    }

    private void refill() {
        long now = clock.millis();
        long passedMillis;
        long lastRefill;
        do {
            lastRefill = this.lastRefill.get();
            passedMillis = now - lastRefill;
            if (passedMillis <= 0) {
                return;
            }
        } while (!this.lastRefill.compareAndSet(lastRefill, now));

        // passedMillis > 0

        long availableNanotokens;
        long updatedNanotokens;
        do {
            availableNanotokens = this.availableNanotokens.get();
            double updatedNanotokensDouble = availableNanotokens + passedMillis * refillNanotokensPerMillis;
            if (updatedNanotokensDouble > capacityNanotokens) {
                updatedNanotokens = capacityNanotokens;
            } else {
                updatedNanotokens = (long) updatedNanotokensDouble;
            }
        } while (!this.availableNanotokens.compareAndSet(availableNanotokens, updatedNanotokens));
    }

    @Override
    public String toString() {
        return "TokenBucketRateLimier{" +
                "capacity=" + (capacityNanotokens * TOKENS_IN_NANOTOKENS) +
                ",available=" + (availableNanotokens.get() * TOKENS_IN_NANOTOKENS) +
                ", lastRefill=" + Instant.ofEpochMilli(lastRefill.get()) +
                '}';
    }
}
