package ru.yandex.solomon.selfmon.jvm;

import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryUsage;
import java.util.Map;

import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.NotificationListener;
import javax.management.openmbean.CompositeData;

import com.sun.management.GarbageCollectionNotificationInfo;

import ru.yandex.monlib.metrics.encode.text.MetricTextEncoder;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
public class JvmAllocations {

    public static void addMetrics(MetricRegistry registry) {
        for (GarbageCollectorMXBean bean : ManagementFactory.getGarbageCollectorMXBeans()) {
            if (!(bean instanceof NotificationEmitter)) {
                continue;
            }

            NotificationEmitter emitter = (NotificationEmitter) bean;
            switch (bean.getName()) {
                case "PS Scavenge":
                case "PS MarkSweep": {
                    GcListener listener = new GcListener("PS Old Gen", "PS Eden Space", registry);
                    emitter.addNotificationListener(listener, null, null);
                    return;
                }
            }
        }
    }

    public static void main(String[] args) {
        addMetrics(MetricRegistry.root());

        System.gc();
        try (MetricTextEncoder e = new MetricTextEncoder(System.out, true)) {
            MetricRegistry.root().accept(0, e);
        }
    }

    public static class GcListener implements NotificationListener {
        private final String oldGen;
        private final String youngGen;
        private final Rate promotionRate;
        private final Rate allocationRate;

        private GcListener(String oldGen, String youngGen, MetricRegistry registry) {
            this.oldGen = oldGen;
            this.youngGen = youngGen;
            this.promotionRate = registry.rate("jvm.gc.promotionRate");
            this.allocationRate = registry.rate("jvm.gc.allocationRate");
        }

        @Override
        public void handleNotification(Notification notification, Object handback) {
            final String type = notification.getType();
            if (!type.equals(GarbageCollectionNotificationInfo.GARBAGE_COLLECTION_NOTIFICATION)) {
                return;
            }

            CompositeData cd = (CompositeData) notification.getUserData();
            GarbageCollectionNotificationInfo info = GarbageCollectionNotificationInfo.from(cd);

            Map<String, MemoryUsage> before = info.getGcInfo().getMemoryUsageBeforeGc();
            Map<String, MemoryUsage> after = info.getGcInfo().getMemoryUsageAfterGc();

            updateAllocation(before, after);
            updatePromotion(before, after);
        }

        private void updateAllocation(Map<String, MemoryUsage> before, Map<String, MemoryUsage> after) {
            final long youngBefore = before.get(youngGen).getUsed();
            final long youngAfter = after.get(youngGen).getUsed();
            final long delta = youngBefore - youngAfter;
            if (delta > 0) {
                allocationRate.add(delta);
            }
        }

        private void updatePromotion(Map<String, MemoryUsage> before, Map<String, MemoryUsage> after) {
            final long oldBefore = before.get(oldGen).getUsed();
            final long oldAfter = after.get(oldGen).getUsed();
            final long delta = oldAfter - oldBefore;
            if (delta > 0L) {
                promotionRate.add(delta);
            }
        }
    }
}
