package ru.yandex.solomon.selfmon.mon;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletionStage;

import javax.annotation.Nullable;

import io.opentracing.Span;
import io.opentracing.Tracer;
import io.opentracing.noop.NoopSpan;
import io.opentracing.tag.Tags;
import io.opentracing.util.GlobalTracer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;

/**
 * @author Sergey Polovko
 */
public class DaoMetricsProxy<TDao> implements InvocationHandler {
    private static final Logger logger = LoggerFactory.getLogger(DaoMetricsProxy.class);

    private final String daoName;
    private final TDao dao;
    private final Map<String, AsyncMetrics> methodMetrics;
    private final AsyncMetrics totalMetrics;

    // for manager ui
    Throwable lastError;
    Instant lastErrorTime = Instant.EPOCH;

    private DaoMetricsProxy(TDao dao, Class<TDao> daoInterface, MetricRegistry registry) {
        this.daoName = daoInterface.getSimpleName();
        this.dao = dao;
        this.methodMetrics = new HashMap<>();
        this.totalMetrics = new AsyncMetrics(registry, "ydb.ops", Labels.of("method", "total"));

        for (Method method : daoInterface.getMethods()) {
            String methodName = getMethodName(method);
            AsyncMetrics metrics = new AsyncMetrics(registry, "ydb.ops", Labels.of("method", methodName));
            if (methodMetrics.put(methodName, metrics) != null) {
                throw new IllegalStateException("Dao " + daoInterface + " has duplicate methods: " + methodName);
            }
        }
    }

    public static String getMethodName(Method method) {
        NameAlias nameAlias = method.getAnnotation(NameAlias.class);
        if (nameAlias != null) {
            return nameAlias.value();
        }
        return method.getName();
    }

    public static <T> T of(T daoImpl, Class<T> daoInterface, MetricRegistry registry) {
        MetricRegistry daoMetrics = registry.subRegistry("dao", daoInterface.getSimpleName());
        DaoMetricsProxy<T> handler = new DaoMetricsProxy<>(daoImpl, daoInterface, daoMetrics);
        //noinspection unchecked
        return (T) Proxy.newProxyInstance(daoInterface.getClassLoader(), new Class[]{daoInterface}, handler);
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        String methodName = getMethodName(method);

        final Span span;
        if (GlobalTracer.isRegistered()) {
            Tracer tracer = GlobalTracer.get();
            span = tracer.buildSpan(daoName + '.' + methodName)
                    .withTag(Tags.SPAN_KIND, Tags.SPAN_KIND_CLIENT)
                    .start();
        } else {
            span = NoopSpan.INSTANCE;
        }

        AsyncMetrics metrics = methodMetrics.get(methodName);
        return invokeMethod(method, args, metrics, span);
    }

    private Object invokeMethod(Method method, Object[] args, @Nullable AsyncMetrics metrics, Span span) throws Throwable {
        long startMillis = System.currentTimeMillis();
        if (metrics != null) {
            metrics.callStarted();
        }
        totalMetrics.callStarted();

        try {
            Object result = method.invoke(dao, args);
            if (result instanceof CompletionStage) {
                CompletionStage<?> cs = (CompletionStage) result;
                cs.whenComplete((r, t) -> {
                    if (t == null) {
                        onOk(metrics, span, startMillis);
                    } else if (t instanceof CancellationException) {
                        // NOP: ignore cancelled futures
                    } else {
                        saveError(method, t);
                        onError(metrics, span, startMillis);
                    }
                });
            } else {
                onOk(metrics, span, startMillis);
            }
            return result;
        } catch (Throwable t) {
            saveError(method, t);
            onError(metrics, span, startMillis);
            throw t;
        }
    }

    private void onOk(@Nullable AsyncMetrics metrics, Span span, long startMillis) {
        long elapsedMillis = System.currentTimeMillis() - startMillis;
        if (metrics != null) {
            metrics.callCompletedOk(elapsedMillis);
        }
        totalMetrics.callCompletedOk(elapsedMillis);
        span.finish();
    }

    private void onError(@Nullable AsyncMetrics metrics, Span span, long startMillis) {
        long elapsedMillis = System.currentTimeMillis() - startMillis;
        if (metrics != null) {
            metrics.callCompletedError(elapsedMillis);
        }
        totalMetrics.callCompletedError(elapsedMillis);
        span.setTag(Tags.ERROR, Boolean.TRUE).finish();
    }

    private void saveError(Method method, Throwable e) {
        lastError = e;
        lastErrorTime = Instant.now();
        logger.warn("{}:{} failed", daoName, getMethodName(method), e);
    }
}
