package ru.yandex.solomon.selfmon.mon;

import java.io.ByteArrayOutputStream;

import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;

import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.encode.MetricEncoder;
import ru.yandex.monlib.metrics.encode.MetricFormat;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;

import static io.netty.handler.codec.http.HttpHeaderNames.ACCEPT;
import static io.netty.handler.codec.http.HttpHeaderNames.ACCEPT_ENCODING;
import static io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;
import static ru.yandex.monlib.metrics.encode.MetricEncoderFactory.createEncoder;

/**
 * @author Vladimir Gordiychuk
 */
public class HttpMetricRegistryEncoder {

    public static ResponseEntity<byte[]> encode(MetricSupplier supplier, ServerHttpRequest request) {
        MetricFormat format = findBestFormat(request.getHeaders().getFirst(ACCEPT.toString()));
        CompressionAlg compression;
        if (format == MetricFormat.SPACK) {
            compression = findBestCompressionAlg(request.getHeaders().getFirst(ACCEPT_ENCODING.toString()));
        } else {
            compression = CompressionAlg.NONE;
        }

        return encode(supplier, format, compression);
    }

    public static ResponseEntity<byte[]> encode(MetricSupplier supplier, ServerHttpRequest request, MetricFormat format, CompressionAlg compression) {
        if (format == MetricFormat.UNKNOWN) {
            format = findBestFormat(request.getHeaders().getFirst(ACCEPT.toString()));
        }
        if (format == MetricFormat.SPACK && compression == CompressionAlg.NONE) {
            compression = findBestCompressionAlg(request.getHeaders().getFirst(ACCEPT_ENCODING.toString()));
        }

        return encode(supplier, format, compression);
    }

    public static ResponseEntity<byte[]> encode(MetricSupplier supplier, MetricFormat format, CompressionAlg compression) {
        ByteArrayOutputStream out = new ByteArrayOutputStream(8 << 10); // 8 KiB
        try (MetricEncoder encoder = createEncoder(out, format, compression)) {
            // TODO: global time must be set by a client
            long tsMillis = 0; // System.currentTimeMillis();
            supplier.supply(tsMillis, encoder);
        } catch (Exception e) {
            throw new IllegalStateException("cannot encode metrics", e);
        }

        return ResponseEntity.ok()
                .header(CONTENT_TYPE.toString(), format.contentType())
                .body(out.toByteArray());
    }

    private static CompressionAlg findBestCompressionAlg(String acceptEncodingHeader) {
        String[] encodings = StringUtils.split(acceptEncodingHeader, ',');
        if (encodings != null) {
            for (String encoding : encodings) {
                CompressionAlg compressionAlg = CompressionAlg.byEncoding(encoding);
                if (compressionAlg != CompressionAlg.NONE) {
                    return compressionAlg;
                }
            }
        }
        return CompressionAlg.NONE;
    }

    private static MetricFormat findBestFormat(String acceptHeader) {
        String[] contentTypes = StringUtils.split(acceptHeader, ',');
        if (contentTypes != null) {
            for (String contentType : contentTypes) {
                MetricFormat format = MetricFormat.byContentType(contentType);
                if (format != MetricFormat.UNKNOWN) {
                    return format;
                }
            }
        }
        return MetricFormat.TEXT;
    }
}
