package ru.yandex.solomon.selfmon.ng;

import java.util.List;
import java.util.function.ToLongFunction;

import io.netty.buffer.ByteBufAllocatorMetric;
import io.netty.buffer.PoolArenaMetric;
import io.netty.buffer.PoolChunkMetric;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.buffer.UnpooledByteBufAllocator;
import io.netty.util.internal.PlatformDependent;

import ru.yandex.monlib.metrics.encode.text.MetricTextEncoder;
import ru.yandex.monlib.metrics.registry.MetricRegistry;


/**
 * @author Sergey Polovko
 */
public class NettyMon {

    public static void addAllocatorMetrics(MetricRegistry registry) {
        pooledAllocatorMetrics(registry);
        unpooleAllocatorMetrics(registry);
    }

    private static void unpooleAllocatorMetrics(MetricRegistry registry) {
        var allocator = UnpooledByteBufAllocator.DEFAULT;
        var metric = allocator.metric();
        registry = registry.subRegistry("allocator", allocatorName(metric.getClass()));

        {
            var heap = registry.subRegistry("memory", "heap");
            heap.lazyGaugeInt64("netty.memory.used", metric::usedHeapMemory);
        }

        {
            var offHeap = registry.subRegistry("memory", "off-heap");
            offHeap.lazyGaugeInt64("netty.memory.used", metric::usedDirectMemory);
        }
    }

    private static void pooledAllocatorMetrics(MetricRegistry registry) {
        var allocator = PooledByteBufAllocator.DEFAULT;
        var metric = allocator.metric();
        registry = registry.subRegistry("allocator", allocatorName(metric.getClass()));

        {
            var heap = registry.subRegistry("memory", "heap");
            heap.lazyGaugeInt64("netty.memory.used", metric::usedHeapMemory);

            heap.lazyRate("netty.memory.arena.allocations", () -> allocations(metric.heapArenas()));
            heap.lazyRate("netty.memory.arena.deallocations", () -> deallocations(metric.heapArenas()));
            heap.lazyGaugeInt64("netty.memory.arena.active_allocations", () -> activeAllocations(metric.heapArenas()));

            heap.lazyGaugeInt64("netty.memory.arena.count", metric::numDirectArenas);
            heap.lazyGaugeInt64("netty.memory.arena.chunk.total_bytes", () -> chunkBytesSize(metric.heapArenas()));
            heap.lazyGaugeInt64("netty.memory.arena.chunk.free_bytes", () -> chunkBytesFree(metric.heapArenas()));
            heap.lazyGaugeInt64("netty.memory.arena.chunk.used_bytes", () -> chunkBytesUsed(metric.heapArenas()));
            heap.lazyGaugeInt64("netty.memory.arena.thread_caches", () -> threadCaches(metric.heapArenas()));
        }

        {
            var offHeap = registry.subRegistry("memory", "off-heap");
            offHeap.lazyGaugeInt64("netty.memory.used", metric::usedDirectMemory);
            offHeap.lazyGaugeInt64("netty.memory.max", PlatformDependent::maxDirectMemory);

            offHeap.lazyRate("netty.memory.arena.allocations", () -> allocations(metric.directArenas()));
            offHeap.lazyRate("netty.memory.arena.deallocations", () -> deallocations(metric.directArenas()));
            offHeap.lazyGaugeInt64("netty.memory.arena.active_allocations", () -> activeAllocations(metric.directArenas()));

            offHeap.lazyGaugeInt64("netty.memory.arena.count", metric::numDirectArenas);
            offHeap.lazyGaugeInt64("netty.memory.arena.chunk.total_bytes", () -> chunkBytesSize(metric.directArenas()));
            offHeap.lazyGaugeInt64("netty.memory.arena.chunk.free_bytes", () -> chunkBytesFree(metric.directArenas()));
            offHeap.lazyGaugeInt64("netty.memory.arena.chunk.used_bytes", () -> chunkBytesUsed(metric.directArenas()));
            offHeap.lazyGaugeInt64("netty.memory.arena.thread_caches", () -> threadCaches(metric.directArenas()));
        }
    }

    private static long chunkBytesSize(List<PoolArenaMetric> arenas) {
        return sumChunks(arenas, PoolChunkMetric::chunkSize);
    }

    private static long chunkBytesFree(List<PoolArenaMetric> arenas) {
        return sumChunks(arenas, PoolChunkMetric::freeBytes);
    }

    private static long chunkBytesUsed(List<PoolArenaMetric> arenas) {
        return chunkBytesSize(arenas) - chunkBytesFree(arenas);
    }

    private static long allocations(List<PoolArenaMetric> metrics) {
        return sum(metrics, PoolArenaMetric::numAllocations);
    }

    private static long deallocations(List<PoolArenaMetric> metrics) {
        return sum(metrics, PoolArenaMetric::numDeallocations);
    }

    private static long activeAllocations(List<PoolArenaMetric> metrics) {
        return sum(metrics, PoolArenaMetric::numActiveAllocations);
    }

    private static long threadCaches(List<PoolArenaMetric> metrics) {
        return sum(metrics, PoolArenaMetric::numThreadCaches);
    }

    private static long sum(List<PoolArenaMetric> metrics, ToLongFunction<PoolArenaMetric> fn) {
        long total = 0;
        for (var metric : metrics) {
            total += fn.applyAsLong(metric);
        }
        return total;
    }

    private static long sumChunks(List<PoolArenaMetric> arenas, ToLongFunction<PoolChunkMetric> fn) {
        long sum = 0;
        for (var arena : arenas) {
            // same logic as for PoolArenaMetric::numActiveBytes
            synchronized (arena) {
                for (var chunks : arena.chunkLists()) {
                    for (var chunk : chunks) {
                        sum += fn.applyAsLong(chunk);
                    }
                }
            }
        }
        return sum;
    }

    private static String allocatorName(Class<? extends ByteBufAllocatorMetric> metricClass) {
        String className = metricClass.getSimpleName();
        if (className.contains("Pooled")) {
            return "pooled";
        } else if (className.contains("Unpooled")) { // because class UnpooledByteBufAllocatorMetric is private
            return "unpooled";
        } else {
            return "unknown";
        }
    }

    // self test
    public static void main(String[] args) {
        MetricRegistry registry = new MetricRegistry();
        addAllocatorMetrics(registry);
        try (MetricTextEncoder e = new MetricTextEncoder(System.out, true)) {
            registry.accept(0, e);
        }
    }
}
