package ru.yandex.solomon.selfmon.ng.linux;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;

import com.google.common.io.Files;

import ru.yandex.solomon.util.parser.ParserSupport;

/**
 * @author Stepan Koltsov
 */
public class ProcPidStat {
    public static final ProcPidStat ZERO = new ProcPidStat();

    private final int pid;
    private final long ppid;
    // Amount of time that this process has been scheduled
    // in user mode, measured in clock ticks
    private final long utime;
    // Amount  of  time  that this process has been scheduled
    // in kernel mode, measured in clock ticks
    private final long stime;
    private final int numThreads;
    private final long startTime;
    // Resident Set Size: number of pages the process has in real memory
    private final long rss;

    private ProcPidStat() {
        this.pid = 0;
        this.ppid = 0;
        this.utime = 0;
        this.stime = 0;
        this.numThreads = 0;
        this.startTime = 0;
        this.rss = 0;
    }

    private ProcPidStat(ParserSupport parser) {
        this.pid = parser.nextUnsignedIntDecimal();
        parser.consume(" (");
        String comm = parser.consumeUntil(')');
        parser.consume(") ");
        char state = parser.consumeChar();
        parser.consume(' ');
        this.ppid = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        long pgrp = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        long session = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        int ttyNr = parser.nextUnsignedIntDecimal();
        parser.consume(' ');
        long tpgid = parser.nextSignedLongDecimal();
        parser.consume(' ');
        long flags = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        long minflt = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        long cminflt = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        long majflt = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        long cmajflt = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        utime = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        stime = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        long cutime = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        long cstime = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        int priority = parser.nextSignedIntDecimal();
        parser.consume(' ');
        int nice = parser.nextSignedIntDecimal();
        parser.consume(' ');
        numThreads = parser.nextUnsignedIntDecimal();
        parser.consume(' ');
        int itrealvalue = parser.nextSignedIntDecimal();
        parser.consume(' ');
        startTime = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        long vsize = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        rss = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        long rsslim = parser.nextUnsignedLongDecimal();
        parser.consume(' ');
        // ...
    }

    public static ProcPidStat parse(String line) {
        try {
            ParserSupport parser = new ParserSupport(line);
            return new ProcPidStat(parser);
        } catch (Exception e) {
            throw new RuntimeException("failed to parse /proc/<pid>/stat text: " + line, e);
        }
    }

    public int getPid() {
        return pid;
    }

    public long getPpid() {
        return ppid;
    }

    public long getUtime() {
        return utime;
    }

    public long getStime() {
        return stime;
    }

    public int getNumThreads() {
        return numThreads;
    }

    public long getStartTime() {
        return startTime;
    }

    public long getRss() {
        return rss;
    }

    public static ProcPidStat readProcSelfStat() {
        File selfStat = new File("/proc/self/stat");
        if (!selfStat.exists()) {
            return ProcPidStat.ZERO;
        }

        try (BufferedReader reader = Files.newReader(selfStat, StandardCharsets.UTF_8)) {
            return parse(reader.readLine());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
