package ru.yandex.solomon.selfmon.failsafe;

import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.selfmon.failsafe.CircuitBreaker.Status;

import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class ExpMovingAverageCircuitBreakerTest {

    private ClockStub clock;

    @Before
    public void setUp() throws Exception {
        this.clock = new ClockStub();
    }

    @Test
    public void initClosed() throws Exception {
        ExpMovingAverageCircuitBreaker circuit = create(0.3);
        assertThat(circuit.getStatus(), equalTo(CircuitBreaker.Status.CLOSED));
    }

    @Test
    public void allowExecuteWhenClosed() throws Exception {
        ExpMovingAverageCircuitBreaker circuit = create(0.5d);
        assertThat(circuit.attemptExecution(), equalTo(true));
        assertThat(circuit.attemptExecution(), equalTo(true));
        clock.passedTime(1, TimeUnit.SECONDS);
        circuit.markSuccess();
        clock.passedTime(5, TimeUnit.SECONDS);
        circuit.markSuccess();
        assertThat(circuit.attemptExecution(), equalTo(true));
    }

    @Test
    public void openWhenFailThresholdReached() throws Exception {
        ExpMovingAverageCircuitBreaker circuit = create(0.3);
        for (int index = 0; index < 100; index++) {
            if (!circuit.attemptExecution()) {
                break;
            }

            if (index % 2 == 0) {
                circuit.markSuccess();
            } else {
                circuit.markFailure();
            }

            clock.passedTime(500, TimeUnit.MILLISECONDS);
        }

        assertThat(circuit.getStatus(), equalTo(CircuitBreaker.Status.OPEN));
    }

    @Test
    public void disableExecuteWhenOpen() throws Exception {
        ExpMovingAverageCircuitBreaker circuit = create(0.5);
        for (int index = 0; index < 100; index++) {
            if (!circuit.attemptExecution()) {
                break;
            }

            if (index % 3 == 0) {
                circuit.markSuccess();
            } else {
                circuit.markFailure();
            }

            clock.passedTime(500, TimeUnit.MILLISECONDS);
        }

        assertThat(circuit.attemptExecution(), equalTo(false));
    }

    @Test
    public void halfOpenAfterResetTime() throws Exception {
        ExpMovingAverageCircuitBreaker circuitBreaker = create(0.5, TimeUnit.MINUTES.toMillis(1));
        for (int index = 0; index < 10; index++) {
            clock.passedTime(1, TimeUnit.SECONDS);
            circuitBreaker.markFailure();
        }

        clock.passedTime(1, TimeUnit.MINUTES);
        assertThat(circuitBreaker.attemptExecution(), equalTo(true));
        assertThat(circuitBreaker.getStatus(), equalTo(CircuitBreaker.Status.HALF_OPEN));
    }

    @Test
    public void halfOpenOnlyOneExecuteAllowed() throws Exception {
        ExpMovingAverageCircuitBreaker circuitBreaker = create(0.5, TimeUnit.MINUTES.toMillis(1));
        for (int index = 0; index < 10; index++) {
            clock.passedTime(1, TimeUnit.SECONDS);
            circuitBreaker.markFailure();
        }

        clock.passedTime(1, TimeUnit.MINUTES);
        assertThat(circuitBreaker.attemptExecution(), equalTo(true));
        clock.passedTime(1, TimeUnit.MINUTES);
        assertThat(circuitBreaker.attemptExecution(), equalTo(false));
    }

    @Test
    public void openAfterFailOnHalfOpen() throws Exception {
        ExpMovingAverageCircuitBreaker circuitBreaker = create(0.5, TimeUnit.MINUTES.toMillis(2));
        for (int index = 0; index < 10; index++) {
            clock.passedTime(1, TimeUnit.SECONDS);
            circuitBreaker.markFailure();
        }

        clock.passedTime(2, TimeUnit.MINUTES);
        assertThat(circuitBreaker.attemptExecution(), equalTo(true));
        clock.passedTime(100, TimeUnit.MILLISECONDS);
        circuitBreaker.markFailure();

        assertThat(circuitBreaker.getStatus(), equalTo(CircuitBreaker.Status.OPEN));
        assertThat(circuitBreaker.attemptExecution(), equalTo(false));
    }

    @Test
    public void closeAfterSuccessOnHalfOpen() throws Exception {
        ExpMovingAverageCircuitBreaker circuitBreaker = create(0.5, TimeUnit.MINUTES.toMillis(3));
        for (int index = 0; index < 10; index++) {
            clock.passedTime(1, TimeUnit.SECONDS);
            circuitBreaker.markFailure();
        }

        clock.passedTime(6, TimeUnit.MINUTES);
        assertThat(circuitBreaker.attemptExecution(), equalTo(true));
        clock.passedTime(200, TimeUnit.MILLISECONDS);
        circuitBreaker.markSuccess();

        assertThat(circuitBreaker.getStatus(), equalTo(CircuitBreaker.Status.CLOSED));
        assertThat(circuitBreaker.attemptExecution(), equalTo(true));
    }

    @Test
    public void multiThread() {
        // It's slow test but it's necessary to test tick interval
        ExpMovingAverageCircuitBreaker circuitBreaker = create(0.5, TimeUnit.MINUTES.toMillis(5));
        IntStream.range(0, 1000)
                .parallel()
                .filter(value -> circuitBreaker.attemptExecution())
                .forEach(index -> {
                    if (index % 100 == 0) {
                        clock.passedTime(1, TimeUnit.SECONDS);
                    }

                    if (index % 3 == 0) {
                        circuitBreaker.markSuccess();
                    } else {
                        circuitBreaker.markFailure();
                    }
                });

        assertThat(circuitBreaker.getStatus(), equalTo(CircuitBreaker.Status.OPEN));
        assertThat(circuitBreaker.attemptExecution(), equalTo(false));

        while (!circuitBreaker.attemptExecution()) {
            long randomDelay = ThreadLocalRandom.current().nextLong(0, 1000);
            clock.passedTime(randomDelay, TimeUnit.MILLISECONDS);
        }
        // reopen
        assertThat(circuitBreaker.getStatus(), equalTo(Status.HALF_OPEN));
        circuitBreaker.markSuccess();
        assertThat(circuitBreaker.getStatus(), equalTo(Status.CLOSED));

        IntStream.range(0, 1000)
                .parallel()
                .filter(index -> circuitBreaker.attemptExecution())
                .forEach(index -> {
                    if (index % 100 == 0) {
                        clock.passedTime(1, TimeUnit.SECONDS);
                    }

                    circuitBreaker.markSuccess();
                });

        assertThat(circuitBreaker.getStatus(), equalTo(CircuitBreaker.Status.CLOSED));
        assertThat(circuitBreaker.attemptExecution(), equalTo(true));
    }

    private ExpMovingAverageCircuitBreaker create(double threshold) {
        return create(threshold, TimeUnit.SECONDS.toMillis(10));
    }

    private ExpMovingAverageCircuitBreaker create(double threshold, long resertTimeoutMillis) {
        return new ExpMovingAverageCircuitBreaker(clock, threshold, resertTimeoutMillis);
    }
}
