package ru.yandex.solomon.selfmon.failsafe;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class InFlightLimiterTest {

    private InFlightLimiter inFlightLimiter;

    @Before
    public void setUp() {
        inFlightLimiter = new DefaultInFlightLimiter(5);
    }

    @Test
    public void passAll() {
        for (int i = 0; i < 5; i++) {
            Assert.assertTrue(inFlightLimiter.tryIncInFlight());
        }
    }

    @Test
    public void rejectRest() {
        for (int i = 0; i < 5; i++) {
            Assert.assertTrue(inFlightLimiter.tryIncInFlight());
        }
        for (int i = 0; i < 50; i++) {
            Assert.assertFalse(inFlightLimiter.tryIncInFlight());
        }
    }

    @Test
    public void allowSome() {
        List<CompletableFuture<Integer>> tasks = IntStream.range(0, 500).mapToObj(i -> new CompletableFuture<Void>()
                .completeOnTimeout(null, ThreadLocalRandom.current().nextInt(10, 100), TimeUnit.MILLISECONDS)
                .thenCompose(aVoid -> {
                    if (!inFlightLimiter.tryIncInFlight()) {
                        return CompletableFuture.completedFuture(0);
                    } else {
                        return new CompletableFuture<Integer>()
                                .completeOnTimeout(100 + i, ThreadLocalRandom.current().nextInt(10, 50), TimeUnit.MILLISECONDS)
                                .whenComplete((integer, throwable) -> inFlightLimiter.decInFlight());
                    }
                }))
            .collect(Collectors.toList());

        var result = CompletableFutures.allOf(tasks).join();

        for (int i = 0; i < 5; i++) {
            Assert.assertTrue(inFlightLimiter.tryIncInFlight());
        }

        long sent = result.stream().filter(x -> x > 0).count();
        long rejected = result.stream().filter(x -> x == 0).count();
        System.out.println("Sent: " + sent + ", rejected: " + rejected);

        Assert.assertTrue(sent >= 5);
        Assert.assertTrue(rejected > 0);
    }
}
