package ru.yandex.solomon.selfmon.failsafe;

import java.util.concurrent.TimeUnit;

import org.junit.Before;
import org.junit.Test;

import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class SimpleCircuitBreakerTest {

    private ClockStub clock;

    @Before
    public void setUp() throws Exception {
        clock = new ClockStub();
    }

    @Test
    public void initClosed() throws Exception {
        SimpleCircuitBreaker circuit = create(10, 1000);
        assertThat(circuit.getStatus(), equalTo(CircuitBreaker.Status.CLOSED));
    }

    @Test
    public void openWhenFailThresholdReached() throws Exception {
        SimpleCircuitBreaker circuit = create(3, 1000);
        circuit.markFailure();
        circuit.markFailure();
        circuit.markFailure();
        assertThat(circuit.getStatus(), equalTo(CircuitBreaker.Status.OPEN));
    }

    @Test
    public void allowExecuteWhenClosed() throws Exception {
        SimpleCircuitBreaker circuit = create(10, 1000);
        assertThat(circuit.attemptExecution(), equalTo(true));
        circuit.markSuccess();
        assertThat(circuit.attemptExecution(), equalTo(true));
    }

    @Test
    public void disableExecuteWhenOpen() throws Exception {
        SimpleCircuitBreaker circuit = create(3, 1000);
        circuit.markFailure();
        circuit.markFailure();
        circuit.markFailure();
        assertThat(circuit.attemptExecution(), equalTo(false));
    }

    @Test
    public void resetFailCountOnSuccess() throws Exception {
        SimpleCircuitBreaker circuit = create(3, 1000);
        circuit.markFailure();
        circuit.markSuccess();
        circuit.markFailure();
        circuit.markSuccess();
        circuit.markFailure();
        assertThat(circuit.getStatus(), equalTo(SimpleCircuitBreaker.Status.CLOSED));
        assertThat(circuit.attemptExecution(), equalTo(true));
    }

    @Test
    public void halfOpenAfterResetTime() throws Exception {
        SimpleCircuitBreaker circuit = create(3, 10);
        circuit.markFailure();
        circuit.markFailure();
        circuit.markFailure();

        clock.passedTime(20, TimeUnit.MILLISECONDS);
        assertThat(circuit.attemptExecution(), equalTo(true));
        assertThat(circuit.getStatus(), equalTo(SimpleCircuitBreaker.Status.HALF_OPEN));
    }

    @Test
    public void closeOnSuccessAfterHalfOpen() throws Exception {
        SimpleCircuitBreaker circuit = create(3, 10);
        circuit.markFailure();
        circuit.markFailure();
        circuit.markFailure();

        clock.passedTime(25, TimeUnit.MILLISECONDS);
        if (circuit.attemptExecution()) {
            circuit.markSuccess();
        }
        assertThat(circuit.getStatus(), equalTo(SimpleCircuitBreaker.Status.CLOSED));
        assertThat(circuit.attemptExecution(), equalTo(true));
    }

    @Test
    public void openOnFailureAfterHalfOpen() throws Exception {
        SimpleCircuitBreaker circuit = create(3, 10);
        circuit.markFailure();
        circuit.markFailure();
        circuit.markFailure();

        clock.passedTime(30, TimeUnit.MILLISECONDS);
        if (circuit.attemptExecution()) {
            circuit.markFailure();
        }
        assertThat(circuit.getStatus(), equalTo(SimpleCircuitBreaker.Status.OPEN));
        assertThat(circuit.attemptExecution(), equalTo(false));
    }

    @Test
    public void onlyOneAttemptOnHalfOpen() throws Exception {
        SimpleCircuitBreaker circuit = create(3, 10);
        circuit.markFailure();
        circuit.markFailure();
        circuit.markFailure();

        clock.passedTime(10, TimeUnit.MILLISECONDS);
        assertThat(circuit.attemptExecution(), equalTo(true));
        assertThat(circuit.attemptExecution(), equalTo(false));
        assertThat(circuit.attemptExecution(), equalTo(false));
    }

    private SimpleCircuitBreaker create(long failureThreshold, long resetTimeoutMillis) {
        return new SimpleCircuitBreaker(clock, 3, 10);
    }
}
