package ru.yandex.solomon.selfmon.failsafe;

import java.time.Duration;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.ut.ManualClock;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class TokenBucketRateLimiterTest {
    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(15, TimeUnit.SECONDS)
            .withLookingForStuckThread(true)
            .build();

    private ManualClock clock;

    @Before
    public void setUp() {
        clock = new ManualClock();
    }

    @Test
    public void immediateSingle() {
        RateLimiter limiter = new TokenBucketRateLimiter(clock, 10, Duration.ofSeconds(5));

        for (int i = 0; i < 10; i++) {
            Assert.assertTrue(limiter.acquire());
        }

        Assert.assertFalse(limiter.acquire());
    }

    @Test
    public void immediateMany() {
        RateLimiter limiter = new TokenBucketRateLimiter(clock, 10, Duration.ofSeconds(5));

        for (int i = 0; i < 3; i++) {
            Assert.assertTrue(limiter.acquire(3));
        }

        Assert.assertFalse(limiter.acquire(3));
    }

    @Test
    public void withSleep() {
        RateLimiter limiter = new TokenBucketRateLimiter(clock, 10, Duration.ofSeconds(5));

        for (int i = 0; i < 3000; i++) {
            Assert.assertTrue(limiter.acquire());
            clock.passedTime(500, TimeUnit.MILLISECONDS);
        }
    }

    @Test
    public void notMoreThan() {
        RateLimiter limiter = new TokenBucketRateLimiter(clock, 10, Duration.ofSeconds(5));

        int oks = 0;

        for (int i = 0; i < 3000; i++) {
            if (limiter.acquire()) {
                oks++;
            }
            clock.passedTime(100, TimeUnit.MILLISECONDS);
        }

        double expected = 3000 * 0.1 * (10.0 / 5.0);

        Assert.assertEquals(expected, oks, 15);
    }

    @Test
    public void multiThread() {
        RateLimiter limiter = new TokenBucketRateLimiter(clock, 10, Duration.ofSeconds(5));

        var executor = Executors.newFixedThreadPool(4);

        final AtomicLong oks = new AtomicLong(0);

        IntStream.range(0, 3000)
                .mapToObj(i -> CompletableFutures.supplyAsync(() -> {
                    boolean ret;
                    if (ret = limiter.acquire()) {
                        oks.getAndIncrement();
                    }
                    clock.passedTime(100, TimeUnit.MILLISECONDS);
                    return ret;
                }, executor))
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOf))
                .join();

        double expected = 3000 * 0.1 * (10.0 / 5.0);

        Assert.assertEquals(expected, oks.get(), 15);
    }
}
