package ru.yandex.solomon.selfmon.trace;

import java.util.concurrent.CompletableFuture;

import io.opentracing.Span;
import io.opentracing.mock.MockTracer;
import org.junit.Before;
import org.junit.Test;

import static org.hamcrest.CoreMatchers.nullValue;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertSame;
import static org.junit.Assume.assumeThat;

/**
 * @author Vladimir Gordiychuk
 */
public class SpanAwareFutureTest {

    private MockTracer tracer;

    @Before
    public void setUp() throws Exception {
        tracer = new MockTracer();
    }

    @Test
    public void wrapNoSpan() {
        var root = new CompletableFuture<String>();
        var span = tracer.activeSpan();
        assertNull(span);
        assumeThat(span, nullValue());

        var wrapped = wrap(root);
        assertSame(root, wrapped);
    }

    @Test
    public void wrapThenCompose() {
        var span = tracer.buildSpan("test").start();
        var root = new CompletableFuture<String>();
        var wrapped = wrap(span, root)
                .thenCompose(v -> {
                    assertEquals("1", v);
                    assertSame(span, tracer.activeSpan());
                    return CompletableFuture.supplyAsync(() -> "2");
                })
                .thenCompose(v -> {
                    assertEquals("2", v);
                    assertSame(span, tracer.activeSpan());
                    return CompletableFuture.completedFuture("3");
                })
                .thenComposeAsync(v -> {
                    assertEquals("3", v);
                    assertSame(span, tracer.activeSpan());
                    return CompletableFuture.completedFuture("4");
                });

        root.complete("1");
        span.finish();
        assertEquals("4", wrapped.join());
    }

    @Test
    public void wrapCompletedThenCompose() {
        var span = tracer.buildSpan("test").start();
        var root = CompletableFuture.completedFuture("1");
        var wrapped = wrap(span, root)
                .thenCompose(v -> {
                    assertEquals("1", v);
                    assertSame(span, tracer.activeSpan());
                    return CompletableFuture.supplyAsync(() -> "2");
                })
                .thenCompose(v -> {
                    assertEquals("2", v);
                    assertSame(span, tracer.activeSpan());
                    return CompletableFuture.completedFuture("3");
                })
                .thenComposeAsync(v -> {
                    assertEquals("3", v);
                    assertSame(span, tracer.activeSpan());
                    return CompletableFuture.completedFuture("4");
                });

        span.finish();
        assertEquals("4", wrapped.join());
    }

    @Test
    public void wrapThenApply() {
        var span = tracer.buildSpan("test").start();
        var root = new CompletableFuture<String>();
        var wrapped = wrap(span, root)
                .thenApply(v -> {
                    assertEquals("1", v);
                    assertSame(span, tracer.activeSpan());
                    return "2";
                })
                .thenApplyAsync(v -> {
                    assertEquals("2", v);
                    assertSame(span, tracer.activeSpan());
                    return "3";
                })
                .thenApplyAsync(v -> {
                    assertEquals("3", v);
                    assertSame(span, tracer.activeSpan());
                    return "4";
                });

        root.complete("1");
        span.finish();
        assertEquals("4", wrapped.join());
    }

    @Test
    public void wrapThenAccept() {
        var span = tracer.buildSpan("test").start();
        var root = new CompletableFuture<String>();
        var wrapped = wrap(span, root)
                .thenAccept(v -> {
                    assertEquals("1", v);
                    assertSame(span, tracer.activeSpan());
                })
                .thenAcceptAsync(v -> {
                    assertSame(span, tracer.activeSpan());
                });

        root.complete("1");
        span.finish();
        wrapped.join();
    }

    @Test
    public void wrapWhenComplete() {
        var span = tracer.buildSpan("test").start();
        var root = new CompletableFuture<String>();
        var wrapped = wrap(span, root)
                .whenComplete((v, e) -> {
                    assertSame(span, tracer.activeSpan());
                })
                .whenCompleteAsync((v, e) -> {
                    assertSame(span, tracer.activeSpan());
                });

        root.complete("1");
        span.finish();
        wrapped.join();
    }

    @Test
    public void wrapTwice() {
        var span = tracer.buildSpan("test").start();
        var root = new CompletableFuture<String>();
        var wrappedOne = wrap(span, root);
        var wrappedTwo = wrap(span, wrappedOne)
                .thenApply(v -> {
                    assertEquals("1", v);
                    assertSame(span, tracer.activeSpan());
                    return "2";
                })
                .thenApplyAsync(v -> {
                    assertEquals("2", v);
                    assertSame(span, tracer.activeSpan());
                    return "3";
                })
                .thenApplyAsync(v -> {
                    assertEquals("3", v);
                    assertSame(span, tracer.activeSpan());
                    return "4";
                });

        root.complete("1");
        span.finish();
        assertEquals("4", wrappedTwo.join());
    }

    private <T> CompletableFuture<T> wrap(Span span, CompletableFuture<T> future) {
        try (var scope = tracer.activateSpan(span)) {
            return SpanAwareFuture.wrap(tracer, future);
        }
    }

    private <T> CompletableFuture<T> wrap(CompletableFuture<T> future) {
        return SpanAwareFuture.wrap(tracer, future);
    }
}
