package ru.yandex.solomon.ctx;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

import com.google.common.net.HostAndPort;
import com.google.common.util.concurrent.MoreExecutors;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import yandex.cloud.auth.api.AsyncCloudAuthClient;
import yandex.cloud.auth.api.CloudAuthConstants;
import yandex.cloud.auth.grpc.internal.AsyncCloudAuthGrpcClientImpl;

import ru.yandex.grpc.utils.client.interceptors.MetricClientInterceptor;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.iam.FakeCloudAuthClient;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * Place for instantiating Cloud specific clients.
 *
 * @author Sergey Polovko
 */
@Configuration
class CloudContext {

    @Bean
    @Primary
    AsyncCloudAuthClient cloudAuthClient(ThreadPoolProvider threads, MetricRegistry registry, Optional<TAuthConfig> authConfig) {
        if (authConfig.isEmpty() || !authConfig.get().hasIamConfig()) {
            return new FakeCloudAuthClient();
        }

        TAuthConfig.TIamConfig iamConfig = authConfig.get().getIamConfig();

        String threadPoolName = iamConfig.getThreadPoolName();
        Executor threadPool = !threadPoolName.isEmpty()
                ? threads.getExecutorService(threadPoolName, "AuthConfig.IamConfig.ThreadPool")
                : MoreExecutors.directExecutor();

        Duration readTimeout = TimeConverter.protoToDuration(iamConfig.getReadTimeout(), Duration.ofSeconds(2));
        HostAndPort address = HostAndPort.fromString(iamConfig.getAccessServiceAddresses());
        ManagedChannel channel = ManagedChannelBuilder.forAddress(address.getHost(), address.getPort())
                .userAgent("Solomon")
                .keepAliveTime(CloudAuthConstants.DEFAULT_KEEP_ALIVE_TIME.toNanos(), TimeUnit.NANOSECONDS)
                .keepAliveTimeout(CloudAuthConstants.DEFAULT_KEEP_ALIVE_TIMEOUT.toNanos(), TimeUnit.NANOSECONDS)
                .keepAliveWithoutCalls(true)
                .enableRetry()
                .intercept(new MetricClientInterceptor(HostUtils.getFqdn(), registry))
                .maxRetryAttempts(CloudAuthConstants.DEFAULT_RETRIES)
                .build();

        return new AsyncCloudAuthGrpcClientImpl(channel, readTimeout, threadPool);
    }
}
