package ru.yandex.solomon.ctx;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import yandex.cloud.auth.api.AsyncCloudAuthClient;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.passport.tvmauth.TvmClient;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.Authenticator;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.iam.IamAuthenticator;
import ru.yandex.solomon.auth.iam.IamTokenContext;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.auth.tvm.TvmAuthenticator;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.util.SolomonEnv;

/**
 * @author Oleg Baryshnikov
 */
@Import({
        CloudContext.class,
        YandexContext.class,
        IamTokenContext.class
})
@ParametersAreNonnullByDefault
public class ServiceAuthContext {

    @Bean
    HttpAuthenticator httpAuthenticator(Authenticator authenticator) {
        return new HttpAuthenticator(authenticator);
    }

    @Bean
    Authenticator authenticator(
        BlackboxClient blackboxClient,
        Optional<TAuthConfig> authConfigO,
        @Qualifier("tvmClient") Optional<TvmClient> tvmClientO,
        Optional<AsyncCloudAuthClient> cloudAuthClientO)
    {
        if (authConfigO.isEmpty()) {
            return Authenticator.anonymous();
        }

        List<Pair<AuthType, Authenticator>> authenticators = new ArrayList<>(3);
        if (authConfigO.get().hasTvmConfig()) {
            TvmClient tvmClient = tvmClientO.orElseThrow();
            authenticators.add(Pair.of(AuthType.TvmService, new TvmAuthenticator(tvmClient, blackboxClient)));
            authenticators.add(Pair.of(AuthType.TvmUser, new TvmAuthenticator(tvmClient, blackboxClient)));
        }

        if (authConfigO.get().hasIamConfig()) {
            authenticators.add(Pair.of(AuthType.IAM, new IamAuthenticator(cloudAuthClientO.orElseThrow())));
        }

        return Authenticator.cache(Authenticator.mux(authenticators));
    }

    @Bean
    InternalAuthorizer authorizer(Optional<TAuthConfig> authConfig) {
        if (authConfig.isEmpty() || !authConfig.get().hasInternalAccess()) {
            if (!SolomonEnv.DEVELOPMENT.isActive()) {
                throw new IllegalStateException("AuthConfig is missing from the config and must be present in " +
                        SolomonEnv.current() + " env");
            }
            return new InternalAuthorizer(Set.of()) {
                @Override
                public CompletableFuture<Account> authorize(AuthSubject subject) {
                    return CompletableFuture.completedFuture(Account.ANONYMOUS);
                }
            };
        }

        var allowed = Set.copyOf(authConfig.get().getInternalAccess().getAllowList());
        return new InternalAuthorizer(allowed);
    }
}
