package ru.yandex.solomon.ctx;

import java.util.Optional;

import com.google.common.primitives.Ints;
import io.netty.util.concurrent.DefaultThreadFactory;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.AsyncHttpClientConfig;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxClientOptions;
import ru.yandex.blackbox.BlackboxClients;
import ru.yandex.passport.tvmauth.BlackboxEnv;
import ru.yandex.passport.tvmauth.NativeTvmClient;
import ru.yandex.passport.tvmauth.TvmApiSettings;
import ru.yandex.passport.tvmauth.TvmClient;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.secrets.SecretProvider;

/**
 * Place for instantiating Yandex specific clients.
 *
 * @author Sergey Polovko
 */
@Configuration
class YandexContext {

    @Bean
    TvmClient tvmClient(Optional<TAuthConfig> authConfig, SecretProvider secrets) {
        if (authConfig.isEmpty() || !authConfig.get().hasTvmConfig()) {
            return null;
        }

        TAuthConfig.TTvmConfig tvmConfig = authConfig.get().getTvmConfig();
        Optional<String> clientSecret = secrets.getSecret(tvmConfig.getSecret());
        try (TvmApiSettings settings = TvmApiSettings.create()
                .setSelfTvmId(tvmConfig.getClientId())
                .enableServiceTicketChecking()
                .enableUserTicketChecking(BlackboxEnv.PROD_YATEAM))
        {
            clientSecret.ifPresent(s -> settings.enableServiceTicketsFetchOptions(s, Ints.toArray(tvmConfig.getDestinationIdsList())));
            return NativeTvmClient.create(settings);
        }
    }

    @Bean
    AsyncHttpClient blackboxHttpClient() {
        AsyncHttpClientConfig httpClientConf = new DefaultAsyncHttpClientConfig.Builder()
                .setThreadFactory(new DefaultThreadFactory("BlackBoxClient", true))
                .build();
        return new DefaultAsyncHttpClient(httpClientConf);
    }

    @Bean
    BlackboxClient yandexAuth(AsyncHttpClient httpClient) {
        var opts = BlackboxClientOptions.newBuilder()
                .setUrl("http://blackbox.yandex-team.ru")
                .setHost("")
                .build();
        return BlackboxClients.create(httpClient, opts);
    }
}
