package ru.yandex.solomon.slog;

import java.util.concurrent.TimeUnit;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;

import ru.yandex.misc.lang.ByteUtils;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.MetricTypes;
import ru.yandex.monlib.metrics.encode.spack.format.MetricValuesType;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.slog.compression.EncodeStream;

import static com.google.protobuf.CodedOutputStream.computeUInt32SizeNoTag;

/**
 * @author Vladimir Gordiychuk
 */
public class LogDataBuilderImpl implements LogDataBuilder {
    private final LogDataHeader header;
    private final RecyclableAggrPoint temp = RecyclableAggrPoint.newInstance();
    private final EncodeStream encoder;
    private boolean build;

    public LogDataBuilderImpl(CompressionAlg alg, ByteBufAllocator allocator, TimePrecision timePrecision, int numId, long commonTsMillis, int stepMillis) {
        this.header = new LogDataHeader(numId, commonTsMillis, stepMillis, timePrecision, alg, DataCodingScheme.POINTS, 0, 0);
        this.encoder = EncodeStream.create(alg, allocator);
        this.encoder.writeHeader(buf -> buf.writeZero(header.size()));
    }

    @Override
    public void onCommonTime(long tsMillis) {
        header.commonTsMillis = tsMillis;
    }

    @Override
    public int onPoint(MetricType type, int flags, AggrPoint point) {
        header.metricsCount++;
        header.pointsCount++;
        return writePoint(type, flags, point);
    }

    @Override
    public int onTimeSeries(MetricType type, int flags, AggrGraphDataArrayList timeSeries) {
        header.metricsCount++;
        header.pointsCount += timeSeries.length();
        return writeTimeSeries(type, flags, timeSeries);
    }

    @Override
    public ByteBuf build() {
        if (build) {
            throw new IllegalStateException("Already build");
        }

        var buffer = encoder.finish();
        int idx = buffer.writerIndex();
        header.writeTo(buffer.resetWriterIndex());
        buffer.writerIndex(idx);
        build = true;
        return buffer;
    }

    @Override
    public void close() {
        encoder.close();
        temp.recycle();
    }

    private int writeTimeSeries(MetricType type, int flags, AggrGraphDataArrayList timeSeries) {
        switch (timeSeries.length()) {
            case 0:
                encoder.writeByte(MetricTypes.pack(type, MetricValuesType.NONE));
                encoder.writeByte(ByteUtils.toByteExact(flags));
                return 2;
            case 1: {
                timeSeries.getDataTo(0, temp);
                return writePoint(type, flags, temp);
            }
            default: {
                return writeMany(type, flags, timeSeries);
            }
        }
    }

    private int writeMany(MetricType type, int flags, AggrGraphDataArrayList timeSeries) {
        encoder.writeByte(MetricTypes.pack(type, MetricValuesType.MANY_WITH_TS));
        encoder.writeByte(ByteUtils.toByteExact(flags));
        encoder.writeVarint32(timeSeries.length());
        int bytes = computeUInt32SizeNoTag(timeSeries.length()) + 2;
        switch (type) {
            case DGAUGE:
                for (int i = 0; i < timeSeries.length(); i++) {
                    bytes += writeTime(timeSeries.getTsMillis(i)) + 8;
                    encoder.writeDouble(timeSeries.getDouble(i));
                    bytes += writeOptional(flags, timeSeries, i);
                }
                break;
            case RATE:
            case IGAUGE:
            case COUNTER:
                for (int i = 0; i < timeSeries.length(); i++) {
                    bytes += writeTime(timeSeries.getTsMillis(i)) + 8;
                    encoder.writeLongLe(timeSeries.getLong(i));
                    bytes += writeOptional(flags, timeSeries, i);
                }
                break;
            case HIST:
            case HIST_RATE:
                for (int i = 0; i < timeSeries.length(); i++) {
                    bytes += writeTime(timeSeries.getTsMillis(i));
                    bytes += writeHistogram(timeSeries.getHistogram(i));
                    bytes += writeOptional(flags, timeSeries, i);
                }
                break;
            case LOG_HISTOGRAM:
                for (int i = 0; i < timeSeries.length(); i++) {
                    bytes += writeTime(timeSeries.getTsMillis(i));
                    bytes += writeLogHistogram(timeSeries.getLogHistogram(i));
                    bytes += writeOptional(flags, timeSeries, i);
                }
                break;
            case DSUMMARY:
                for (int i = 0; i < timeSeries.length(); i++) {
                    bytes += writeTime(timeSeries.getTsMillis(i));
                    bytes += writeDSummary(timeSeries.getSummaryDouble(i));
                    bytes += writeOptional(flags, timeSeries, i);
                }
                break;
            case ISUMMARY:
                for (int i = 0; i < timeSeries.length(); i++) {
                    bytes += writeTime(timeSeries.getTsMillis(i));
                    bytes += writeISummary(timeSeries.getSummaryInt64(i));
                    bytes += writeOptional(flags, timeSeries, i);
                }
                break;
            default:
                throw new UnsupportedOperationException("Unsupported metric type: " + type);
        }
        return bytes;
    }

    private int writePoint(MetricType type, int flags, AggrPoint point) {
        int bytes = 2;
        if (point.tsMillis == 0 || point.tsMillis == header.commonTsMillis) {
            encoder.writeByte(MetricTypes.pack(type, MetricValuesType.ONE_WITHOUT_TS));
            encoder.writeByte(ByteUtils.toByteExact(flags));
        } else {
            encoder.writeByte(MetricTypes.pack(type, MetricValuesType.ONE_WITH_TS));
            encoder.writeByte(ByteUtils.toByteExact(flags));
            bytes += writeTime(point.tsMillis);
        }
        switch (type) {
            case DGAUGE:
                encoder.writeDouble(point.valueNum);
                bytes += 8;
                break;
            case RATE:
            case IGAUGE:
            case COUNTER:
                encoder.writeLongLe(point.longValue);
                bytes += 8;
                break;
            case HIST:
            case HIST_RATE:
                bytes += writeHistogram(point.histogram);
                break;
            case DSUMMARY:
                bytes += writeDSummary(point.summaryDouble);
                break;
            case ISUMMARY:
                bytes += writeISummary(point.summaryInt64);
                break;
            case LOG_HISTOGRAM:
                bytes += writeLogHistogram(point.logHistogram);
                break;
            default:
                throw new UnsupportedOperationException("Unsupported metric type: " + type);
        }
        bytes += writeOptional(flags, point);
        return bytes;
    }

    private int writeOptional(int flags, AggrPoint point) {
        if (flags == 0) {
            return 0;
        }

        int bytes = 0;
        if (LogFlags.COUNT.isInSet(flags)) {
            encoder.writeLongLe(point.count);
            bytes += 8;
        }

        if (LogFlags.DENOM.isInSet(flags)) {
            encoder.writeLongLe(point.valueDenom);
            bytes += 8;
        }
        return bytes;
    }

    private int writeOptional(int flags, AggrGraphDataArrayList timeSeries, int index) {
        if (flags == 0) {
            return 0;
        }

        int bytes = 0;
        if (LogFlags.COUNT.isInSet(flags)) {
            encoder.writeLongLe(timeSeries.getCount(index));
            bytes += 8;
        }

        if (LogFlags.DENOM.isInSet(flags)) {
            encoder.writeLongLe(timeSeries.getDenom(index));
            bytes += 8;
        }
        return bytes;
    }

    private int writeTime(long tsMillis) {
        if (header.timePrecision == TimePrecision.SECONDS) {
            encoder.writeIntLe((int) TimeUnit.MILLISECONDS.toSeconds(tsMillis));
            return 4;
        } else {
            encoder.writeLongLe(tsMillis);
            return 8;
        }
    }

    private int writeHistogram(Histogram histogram) {
        encoder.writeVarint32(histogram.count());
        for (int index = 0; index < histogram.count(); index++) {
            encoder.writeDouble(histogram.upperBound(index));
        }
        for (int index = 0; index < histogram.count(); index++) {
            encoder.writeLongLe(histogram.value(index));
        }
        return computeUInt32SizeNoTag(histogram.count()) + (histogram.count() * 16);
    }

    private int writeDSummary(SummaryDoubleSnapshot summary) {
        encoder.writeLongLe(summary.getCount());
        encoder.writeDouble(summary.getSum());
        encoder.writeDouble(summary.getMin());
        encoder.writeDouble(summary.getMax());
        encoder.writeDouble(summary.getLast());
        return 40;
    }

    private int writeISummary(SummaryInt64Snapshot summary) {
        encoder.writeLongLe(summary.getCount());
        encoder.writeLongLe(summary.getSum());
        encoder.writeLongLe(summary.getMin());
        encoder.writeLongLe(summary.getMax());
        encoder.writeLongLe(summary.getLast());
        return 40;
    }

    private int writeLogHistogram(LogHistogram histogram) {
        encoder.writeDouble(histogram.getBase());
        encoder.writeLongLe(histogram.getCountZero());
        encoder.writeVarint32(histogram.getStartPower());
        encoder.writeVarint32(histogram.countBucket());
        for (int index = 0; index < histogram.countBucket(); index++) {
            encoder.writeDouble(histogram.getBucketValue(index));
        }

        return computeUInt32SizeNoTag(histogram.countBucket())
            + computeUInt32SizeNoTag(histogram.getStartPower())
            + 16
            + (histogram.countBucket() * 8);
    }
}
