package ru.yandex.solomon.slog;

import io.netty.buffer.ByteBuf;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;

/**
 * @author Vladimir Gordiychuk
 */
public class LogDataHeader {
    /**
     * defines how many bytes used for header by current implementation
     */
    private static final short HEADER_SIZE = 32;

    /**
     * expected format magic number
     */
    public static final short VALID_MAGIC = 0x444C; // "LD" in LE-order

    public final LogDataVersion version;
    public final int numId;
    public long commonTsMillis;
    public final int stepMillis;
    public final TimePrecision timePrecision;
    public final CompressionAlg compressionAlg;
    public final DataCodingScheme dataCodingScheme;
    public int metricsCount;
    public int pointsCount;

    public LogDataHeader(
        int numId,
        long commonTsMillis,
        int stepMillis,
        TimePrecision timePrecision,
        CompressionAlg compressionAlg,
        DataCodingScheme dataCodingScheme,
        int metricsCount,
        int pointsCount)
    {
        this.version = LogDataVersion.CURRENT;
        this.numId = numId;
        this.commonTsMillis = commonTsMillis;
        this.stepMillis = stepMillis;
        this.timePrecision = timePrecision;
        this.compressionAlg = compressionAlg;
        this.dataCodingScheme = dataCodingScheme;
        this.metricsCount = metricsCount;
        this.pointsCount = pointsCount;
    }

    public LogDataHeader(ByteBuf in) {
        if (in.readableBytes() < HEADER_SIZE) {
            throw new CorruptedBinaryDataRuntimeException(
                "not enough bytes in buffer to read header, " +
                    "need at least: " + HEADER_SIZE + ", but got: " + in.readableBytes());
        }

        final int magic = in.readShortLE();
        if (magic != VALID_MAGIC) {
            throw new CorruptedBinaryDataRuntimeException("invalid magic, expected " +
                Integer.toString(VALID_MAGIC, 16) + ", got " + Integer.toString(magic, 16));
        }

        in.skipBytes(1);
        this.version = LogDataVersion.valueOf(in.readByte());
        this.numId = in.readIntLE();
        this.commonTsMillis = in.readLongLE();
        this.stepMillis = in.readIntLE();
        this.metricsCount = in.readIntLE();
        this.pointsCount = in.readIntLE();
        this.timePrecision = TimePrecision.valueOf(in.readByte());
        this.compressionAlg = CompressionAlg.valueOf(in.readByte());
        this.dataCodingScheme = DataCodingScheme.valueOf(in.readByte());
        in.skipBytes(1);
    }

    public void writeTo(ByteBuf buffer) {
        writeTo(buffer, metricsCount, pointsCount);
    }

    public void writeTo(ByteBuf buffer, int metricsCount, int pointsCount) {
        buffer
            .writeShortLE(VALID_MAGIC) // 2
            .writeZero(1) // 3
            .writeByte(version.number) // 4
            .writeIntLE(numId) // 8
            .writeLongLE(commonTsMillis) // 16
            .writeIntLE(stepMillis) // 20
            .writeIntLE(metricsCount) // 24
            .writeIntLE(pointsCount) // 28
            .writeByte(timePrecision.value()) // 29
            .writeByte(compressionAlg.value()) // 30
            .writeByte(dataCodingScheme.num) // 31
            .writeZero(1); // 32
    }

    public void setMetricsCount(int idx, ByteBuf buffer, int count) {
        buffer.setIntLE(idx + 20, count);
    }

    public void setPointsCount(int idx, ByteBuf buffer, int count) {
        buffer.setIntLE(idx + 24, count);
    }

    public int size() {
        return HEADER_SIZE;
    }

    @Override
    public String toString() {
        return "LogDataHeader{" +
            "version=" + version +
            ", numId=" + Integer.toUnsignedLong(numId) +
            ", commonTsMillis=" + commonTsMillis +
            ", stepMillis=" + stepMillis +
            ", timePrecision=" + timePrecision +
            ", compressionAlg=" + compressionAlg +
            ", dataCodingScheme=" + dataCodingScheme +
            ", metricsCount=" + metricsCount +
            ", pointsCount=" + pointsCount +
            '}';
    }
}
