package ru.yandex.solomon.slog;

import java.util.NoSuchElementException;

import javax.annotation.WillNotClose;

import io.netty.buffer.ByteBuf;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.MetricTypes;
import ru.yandex.monlib.metrics.encode.spack.format.MetricValuesType;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.model.type.SummaryDouble;
import ru.yandex.solomon.model.type.SummaryInt64;
import ru.yandex.solomon.slog.compression.DecodeStream;

/**
 * @author Vladimir Gordiychuk
 */
public class LogDataIteratorImpl implements LogDataIterator {
    private static final Logger logger = LoggerFactory.getLogger(LogDataIteratorImpl.class);
    private final LogDataHeader header;
    private final DecodeStream in;
    private int pos;
    private final RecyclableAggrPoint tmp;

    public LogDataIteratorImpl(LogDataHeader header, @WillNotClose ByteBuf buffer) {
        this.header = header;
        this.in = DecodeStream.create(header.compressionAlg, buffer.retain());
        this.tmp = RecyclableAggrPoint.newInstance();
    }

    @Override
    public int getNumId() {
        return header.numId;
    }

    @Override
    public int getPointsCount() {
        return header.pointsCount;
    }

    @Override
    public int getMetricsCount() {
        return header.metricsCount;
    }

    @Override
    public boolean hasNext() {
        return pos < header.metricsCount;
    }

    @Override
    public void next(MetricArchiveMutable archive) {
        if (pos++ >= header.metricsCount) {
            throw new NoSuchElementException();
        }

        // (1) types byte
        final byte typesByte = in.readByte();
        final MetricType type = MetricTypes.metricType(typesByte);
        if (type == MetricType.UNKNOWN) {
            throw new CorruptedBinaryDataRuntimeException("unknown metric type #" + pos);
        }

        var protoType = MetricTypeConverter.toNotNullProto(type);
        if (archive.getRecordCount() > 0 && !MetricTypeTransfers.isAvailableTransfer(archive.getType(), protoType)) {
            logger.warn("Not able convert from {} to {} for archive {}", archive.getType(), protoType, archive.header());
            archive = new MetricArchiveMutable();
        }

        archive.setOwnerShardId(header.numId);
        archive.setType(protoType);
        final MetricValuesType metricValuesType = MetricTypes.valuesType(typesByte);

        // (2) flags byte
        final byte flags = in.readByte();
        // (3) values
        switch (metricValuesType) {
            case ONE_WITHOUT_TS: {
                processValueOne(archive, type, flags, header.commonTsMillis);
                return;
            }
            case ONE_WITH_TS: {
                long ts = readTsMillisOrUseCommon();
                processValueOne(archive, type, flags, ts);
                return;
            }
            case MANY_WITH_TS: {
                processValueMany(archive, type, flags);
                return;
            }
            case NONE:
                // skip metrics without values
                return;
            default:
                throw new CorruptedBinaryDataRuntimeException("invalid metric value type: " + metricValuesType);
        }
    }

    private long readTsMillisOrUseCommon() {
        long tsMillis = readTsMillis();
        if (tsMillis == 0) {
            tsMillis = header.commonTsMillis;
        }
        return tsMillis;
    }

    private long readTsMillis() {
        return header.timePrecision == TimePrecision.SECONDS
            ? (long) in.readIntLe() * 1000
            : in.readLongLe();
    }

    private void processValueOne(MetricArchiveMutable archive, MetricType type, int flags, long tsMillis) {
        tmp.columnSet = 0;
        tmp.setTsMillis(tsMillis);
        switch (type) {
            case DGAUGE: {
                tmp.setValue(in.readDoubleLe());
                break;
            }
            case RATE:
            case IGAUGE:
            case COUNTER: {
                tmp.setLongValue(in.readLongLe());
                break;
            }
            case HIST_RATE:
            case HIST: {
                tmp.setHistogram(Histogram.orNew(tmp.histogram));
                readHistogram(tmp.histogram);
                break;
            }
            case LOG_HISTOGRAM: {
                tmp.setLogHistogram(LogHistogram.orNew(tmp.logHistogram));
                readLogHistogram(tmp.logHistogram);
                break;
            }
            case DSUMMARY: {
                var summary = SummaryDouble.orNew(tmp.summaryDouble);
                tmp.setSummaryDouble(summary);
                readSummaryDouble(summary);
                break;
            }
            case ISUMMARY: {
                var summary = SummaryInt64.orNew(tmp.summaryInt64);
                tmp.setSummaryInt64(summary);
                readSummaryInt64(summary);
                break;
            }
            default:
                throw new CorruptedBinaryDataRuntimeException("Unsupported metric type: " + type);
        }
        readOptional(tmp, flags);
        tmp.setStepMillis(header.stepMillis);
        archive.addRecord(tmp);
    }

    private void processValueMany(MetricArchiveMutable archive, MetricType type, int flags) {
        int count = in.readVarint32();
        tmp.columnSet = 0;
        tmp.setStepMillis(header.stepMillis);
        switch (type) {
            case DGAUGE:
                for (int i = 0; i < count; i++) {
                    tmp.setTsMillis(readTsMillis());
                    tmp.setValue(in.readDoubleLe());
                    readOptional(tmp, flags);
                    archive.addRecord(tmp);
                }
                break;
            case RATE:
            case IGAUGE:
            case COUNTER:
                for (int i = 0; i < count; i++) {
                    tmp.setTsMillis(readTsMillis());
                    tmp.setLongValue(in.readLongLe());
                    readOptional(tmp, flags);
                    archive.addRecord(tmp);
                }
                break;
            case HIST:
            case HIST_RATE:
            {
                var hist = Histogram.orNew(tmp.histogram);
                tmp.setHistogram(hist);
                for (int i = 0; i < count; i++) {
                    hist.reset();
                    tmp.setTsMillis(readTsMillis());
                    readHistogram(hist);
                    readOptional(tmp, flags);
                    archive.addRecord(tmp);
                }
                break;
            }
            case LOG_HISTOGRAM:
            {
                var histogram = LogHistogram.orNew(tmp.logHistogram);
                tmp.setLogHistogram(histogram);
                for (int i = 0; i < count; i++) {
                    histogram.reset();
                    tmp.setTsMillis(readTsMillis());
                    readLogHistogram(histogram);
                    histogram.truncateBuckets();
                    readOptional(tmp, flags);
                    archive.addRecord(tmp);
                }
                break;
            }
            case DSUMMARY:
            {
                var summary = SummaryDouble.orNew(tmp.summaryDouble);
                tmp.setSummaryDouble(summary);
                for (int i = 0; i < count; i++) {
                    summary.reset();
                    tmp.setTsMillis(readTsMillis());
                    readSummaryDouble(summary);
                    readOptional(tmp, flags);
                    archive.addRecord(tmp);
                }
                break;
            }
            case ISUMMARY:
            {
                var summary = SummaryInt64.orNew(tmp.summaryInt64);
                tmp.setSummaryInt64(summary);
                for (int i = 0; i < count; i++) {
                    summary.reset();
                    tmp.setTsMillis(readTsMillis());
                    readSummaryInt64(summary);
                    readOptional(tmp, flags);
                    archive.addRecord(tmp);
                }
                break;
            }
            default:
                throw new CorruptedBinaryDataRuntimeException("Unsupported metric type: " + type);
        }
    }

    private void readHistogram(Histogram histogram) {
        int count = in.readVarint32();
        for (int index = 0; index < count; index++) {
            histogram.setUpperBound(index, in.readDoubleLe());
        }
        for (int index = 0; index < count; index++) {
            histogram.setBucketValue(index, in.readLongLe());
        }
    }

    private void readLogHistogram(LogHistogram histogram) {
        histogram.setBase(in.readDoubleLe());
        histogram.setCountZero(in.readLongLe());
        histogram.setStartPower(in.readVarint32());
        var count = in.readVarint32();
        for (int index = 0; index < count; index++) {
            histogram.addBucket(in.readDoubleLe());
        }
    }

    private void readSummaryDouble(SummaryDouble summary) {
        summary.setCount(in.readLongLe());
        summary.setSum(in.readDoubleLe());
        summary.setMin(in.readDoubleLe());
        summary.setMax(in.readDoubleLe());
        summary.setLast(in.readDoubleLe());
    }

    private void readSummaryInt64(SummaryInt64 summary) {
        summary.setCount(in.readLongLe());
        summary.setSum(in.readLongLe());
        summary.setMin(in.readLongLe());
        summary.setMax(in.readLongLe());
        summary.setLast(in.readLongLe());
    }

    private void readOptional(AggrPoint point, int flags) {
        if (flags == 0) {
            return;
        }

        if (LogFlags.COUNT.isInSet(flags)) {
            if (LogFlags.MERGE.isInSet(flags)) {
                point.setMerge(true);
            }
            point.setCount(in.readLongLe());
        }

        if (LogFlags.DENOM.isInSet(flags)) {
            point.valueDenom = in.readLongLe();
            if (!ValueColumn.isValidDenom(point.valueDenom)) {
                throw new IllegalArgumentException("Invalid denom: " + point.valueDenom);
            }
        } else {
            point.valueDenom = ValueColumn.DEFAULT_DENOM;
        }
    }

    @Override
    public void close() {
        in.close();
        tmp.recycle();
    }
}
