package ru.yandex.solomon.slog;

import javax.annotation.Nullable;

import io.netty.buffer.ByteBuf;

/**
 * @author Vladimir Gordiychuk
 */
public final class Logs {
    private Logs() {
    }

    /**
     * @param left take ownership
     * @param right take ownership
     */
    public static Log combineResolvedLog(@Nullable Log left, @Nullable Log right) {
        if (left == null) {
            return right;
        } else if (right == null) {
            return left;
        }

        var meta = combineResolvedMeta(left.meta, right.meta);
        var data = combineDataOrSnapshot(left.data, right.data);
        return new Log(left.numId, meta, data);
    }

    // TODO: drop it after migrate on snapshots (gordiychuk@)
    private static ByteBuf combineDataOrSnapshot(ByteBuf left, ByteBuf right) {
        int magic = left.getShortLE(left.readerIndex());
        switch (magic) {
            case LogDataHeader.VALID_MAGIC:
                return combineData(left, right);
            case SnapshotLogDataHeader.VALID_MAGIC:
                return combineSnapshotData(left, right);
            default:
                throw new IllegalArgumentException("Invalid magic: " + Integer.toString(magic, 16));
        }
    }

    /**
     * @param left take ownership
     * @param right take ownership
     */
    public static ByteBuf combineData(ByteBuf left, ByteBuf right) {
        int readerIdx = left.readerIndex();
        var leftHeader = new LogDataHeader(left);
        var rightHeader = new LogDataHeader(right);
        if (leftHeader.version != rightHeader.version
                || leftHeader.compressionAlg != rightHeader.compressionAlg
                || leftHeader.numId != rightHeader.numId
                || leftHeader.commonTsMillis != rightHeader.commonTsMillis
                || leftHeader.stepMillis != rightHeader.stepMillis
                || leftHeader.timePrecision != rightHeader.timePrecision
        )
        {
            throw new IllegalArgumentException("ResolvedMetaLog header mismatch " + left + " != " + right);
        }

        left.readerIndex(readerIdx);
        leftHeader.setMetricsCount(readerIdx, left, leftHeader.metricsCount + rightHeader.metricsCount);
        leftHeader.setPointsCount(readerIdx, left, leftHeader.pointsCount + rightHeader.pointsCount);
        var result = left.alloc().compositeBuffer(2);
        result.addComponent(true, left);
        result.addComponent(true, right);
        return result;
    }

    /**
     * @param left take ownership
     * @param right take ownership
     */
    public static ByteBuf combineSnapshotData(ByteBuf left, ByteBuf right) {
        int readerIdx = left.readerIndex();
        var leftHeader = new SnapshotLogDataHeader(left);
        var rightHeader = new SnapshotLogDataHeader(right);
        if (leftHeader.version != rightHeader.version
                || leftHeader.compressionAlg != rightHeader.compressionAlg
                || leftHeader.numId != rightHeader.numId)
        {
            throw new IllegalArgumentException("SnapshotLogDataHeader mismatch " + left + " != " + right);
        }

        left.readerIndex(readerIdx);
        leftHeader.setMetricsCount(readerIdx, left, leftHeader.metricsCount + rightHeader.metricsCount);
        leftHeader.setPointsCount(readerIdx, left, leftHeader.pointsCount + rightHeader.pointsCount);
        var result = left.alloc().compositeBuffer(2);
        result.addComponent(true, left);
        result.addComponent(true, right);
        return result;
    }

    /**
     * @param left take ownership
     * @param right take ownership
     */
    public static ByteBuf combineResolvedMeta(ByteBuf left, ByteBuf right) {
        int readerIdx = left.readerIndex();
        var leftHeader = new ResolvedLogMetaHeader(left);
        var rightHeader = new ResolvedLogMetaHeader(right);
        if (leftHeader.version != rightHeader.version
                || leftHeader.compressionAlg != rightHeader.compressionAlg
                || leftHeader.numId != rightHeader.numId
                || leftHeader.producerId != rightHeader.producerId
                || leftHeader.producerSeqNo != rightHeader.producerSeqNo)
        {
            throw new IllegalArgumentException("ResolvedMetaLog header mismatch " + left + " != " + right);
        }

        left.readerIndex(readerIdx);
        leftHeader.setMetricsCount(readerIdx, left, leftHeader.metricsCount + rightHeader.metricsCount);
        leftHeader.setPointsCount(readerIdx, left, leftHeader.pointsCount + rightHeader.pointsCount);
        var result = left.alloc().compositeBuffer(2);
        result.addComponent(true, left);
        result.addComponent(true, right);
        return result;
    }
}
