package ru.yandex.solomon.slog;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;

import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;

/**
 * @author Vladimir Gordiychuk
 */
public class LogsIndexSerializer {
    /**
     * expected format magic number
     */
    private static final short VALID_MAGIC = 0x494C; // "LI" in LE-order

    /**
     * versions
     */
    public static final short VERSION_1_0 = 0x0100;
    private static final short CURRENT_VERSION = VERSION_1_0; // msb - major, lsb - minor

    public static ByteBuf serialize(ByteBufAllocator allocator, LogsIndex index) {
        var buffer = allocator.buffer(12 + (index.getSize() * 12));
        return serialize(buffer, index);
    }

    public static ByteBuf serialize(ByteBuf buffer, LogsIndex index) {
        buffer.writeShortLE(VALID_MAGIC);
        buffer.writeShortLE(CURRENT_VERSION);
        buffer.writeIntLE(index.getSize());
        for (int idx = 0; idx < index.getSize(); idx++) {
            buffer.writeIntLE(index.getNumId(idx));
            buffer.writeIntLE(index.getMetaSize(idx));
            buffer.writeIntLE(index.getDataSize(idx));
        }
        return buffer;
    }

    public static LogsIndex deserialize(ByteBuf in) {
        try {
            final int magic = in.readShortLE();
            if (magic != VALID_MAGIC) {
                throw new CorruptedBinaryDataRuntimeException("invalid magic, expected " +
                    Integer.toString(VALID_MAGIC, 16) + ", got " + Integer.toString(magic, 16));
            }

            var version = in.readShortLE();
            if (version > CURRENT_VERSION) {
                throw new UnsupportedOperationException("Unsupported index version, " + version);
            }

            var size = in.readIntLE();
            var result = new LogsIndex(size);
            for (int idx = 0; idx < size; idx++) {
                int numId = in.readIntLE();
                int metaSize = in.readIntLE();
                int dataSize = in.readIntLE();
                result.add(numId, metaSize, dataSize);
            }

            return result;
        } finally {
            in.release();
        }
    }
}
