package ru.yandex.solomon.slog;

import io.netty.buffer.ByteBuf;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;
import ru.yandex.stockpile.api.EDecimPolicy;

/**
 * @author Vladimir Gordiychuk
 */
public class ResolvedLogMetaHeader {
    /**
     * defines how many bytes used for header by current implementation
     */
    private static final short HEADER_SIZE = 32;
    private static final short SKIP_BYTES = 2;

    /**
     * expected format magic number
     */
    private static final short VALID_MAGIC = 0x4D52; // "RM" in LE-order

    public final ResolvedLogMetaVersion version;
    public final int numId;
    public final CompressionAlg compressionAlg;
    public EDecimPolicy decimPolicy = EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS;
    public int producerId;
    public long producerSeqNo;
    public int metricsCount;
    public int pointsCount;

    public ResolvedLogMetaHeader(int numId, CompressionAlg compressionAlg)
    {
        this.version = ResolvedLogMetaVersion.CURRENT;
        this.numId = numId;
        this.compressionAlg = compressionAlg;
    }

    public ResolvedLogMetaHeader(ByteBuf in) {
        if (in.readableBytes() < HEADER_SIZE) {
            throw new CorruptedBinaryDataRuntimeException(
                "not enough bytes in buffer to read header, " +
                    "need at least: " + HEADER_SIZE + ", but got: " + in.readableBytes());
        }

        final int magic = in.readShortLE();
        if (magic != VALID_MAGIC) {
            throw new CorruptedBinaryDataRuntimeException("invalid magic, expected " +
                Integer.toString(VALID_MAGIC, 16) + ", got " + Integer.toString(magic, 16));
        }

        in.skipBytes(1);
        this.version = ResolvedLogMetaVersion.valueOf(in.readByte());
        this.numId = in.readIntLE();
        this.producerId = in.readIntLE();
        this.producerSeqNo = in.readLongLE();
        this.metricsCount = in.readIntLE();
        this.pointsCount = in.readIntLE();
        this.compressionAlg = CompressionAlg.valueOf(in.readByte());
        this.decimPolicy = EDecimPolicy.forNumber(in.readByte());
        in.skipBytes(SKIP_BYTES);
    }

    public static int producerId(ByteBuf buffer) {
        return buffer.getIntLE(buffer.readerIndex() + 8);
    }

    public static long producerSeqNo(ByteBuf buffer) {
        return buffer.getLongLE(buffer.readerIndex() + 12);
    }

    public ResolvedLogMetaHeader setDecimPolicy(EDecimPolicy decimPolicy) {
        this.decimPolicy = decimPolicy;
        return this;
    }

    public ResolvedLogMetaHeader setProducerId(int producerId) {
        this.producerId = producerId;
        return this;
    }

    public ResolvedLogMetaHeader setProducerSeqNo(long seqNo) {
        this.producerSeqNo = seqNo;
        return this;
    }

    public void writeTo(ByteBuf buffer) {
        buffer
            .writeShortLE(VALID_MAGIC) // 2
            .writeZero(1) // 3
            .writeByte(version.number) // 4
            .writeIntLE(numId) // 8
            .writeIntLE(producerId) // 12
            .writeLongLE(producerSeqNo) // 20
            .writeIntLE(metricsCount) // 24
            .writeIntLE(pointsCount) // 28
            .writeByte(compressionAlg.value()) // 29
            .writeByte(decimPolicy.getNumber()) // 30
            .writeZero(SKIP_BYTES); // 32
    }

    public void setMetricsCount(int idx, ByteBuf buffer, int count) {
        buffer.setIntLE(idx + 20, count);
    }

    public void setPointsCount(int idx, ByteBuf buffer, int count) {
        buffer.setIntLE(idx + 24, count);
    }

    public int size() {
        return HEADER_SIZE;
    }

    @Override
    public String toString() {
        return "ResolvedLogMetaHeader{" +
            "version=" + version +
            ", numId=" + Integer.toUnsignedLong(numId) +
            ", compressionAlg=" + compressionAlg +
            ", producerId=" + producerId +
            ", producerSeqNo=" + producerSeqNo +
            ", decimPolicy=" + decimPolicy +
            ", metricsCount=" + metricsCount +
            ", pointsCount=" + pointsCount +
            '}';
    }
}
