package ru.yandex.solomon.slog;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.CompositeByteBuf;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.MetricTypes;
import ru.yandex.monlib.metrics.encode.spack.format.MetricValuesType;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.slog.compression.EncodeStream;

/**
 * @author Vladimir Gordiychuk
 */
public class ResolvedLogMetricsBuilderImpl implements ResolvedLogMetricsBuilder {

    private final ByteBufAllocator allocator;
    private final CompressionAlg alg;

    private final int numId;
    private int metrics;

    private final EncodeStream meta;
    private final StringPoolEncoder valuesPool;
    private final StringPoolEncoder keysPool;
    private boolean build = false;

    public ResolvedLogMetricsBuilderImpl(int numId, CompressionAlg alg, ByteBufAllocator allocator) {
        this.allocator = allocator;
        this.alg = alg;
        this.numId = numId;
        this.meta = EncodeStream.create(alg, allocator);
        this.keysPool = new StringPoolEncoder(alg, allocator);
        this.valuesPool = new StringPoolEncoder(alg, allocator);
    }

    @Override
    public void onMetric(MetricType type, Labels labels, int shardId, long localId) {
        this.metrics++;
        this.meta.writeByte(MetricTypes.pack(type, MetricValuesType.NONE));
        this.writeLabels(labels);
        this.meta.writeVarint32(shardId);
        this.meta.writeLongLe(localId);
    }

    private void writeLabels(Labels labels) {
        meta.writeVarint32(labels.size());
        labels.forEach(this::writeLabel);
    }

    private void writeLabel(Label label) {
        meta.writeVarint32(keysPool.writeString(label.getKey()));
        meta.writeVarint32(valuesPool.writeString(label.getValue()));
    }

    @Override
    public ByteBuf build() {
        if (build) {
            throw new IllegalStateException("Already build");
        }

        var keysPool = this.keysPool.finish();
        var valuesPool = this.valuesPool.finish();
        var meta = this.meta.finish();

        var header = new ResolvedLogMetricsHeader(
            numId,
            alg,
            this.keysPool.getBytesSize(),
            this.valuesPool.getBytesSize(),
                metrics);

        ByteBuf headerBuffer = allocator.buffer(header.size(), header.size());
        header.writeTo(headerBuffer);
        CompositeByteBuf compositeMeta = allocator.compositeBuffer();
        compositeMeta.addComponent(true, headerBuffer);
        compositeMeta.addComponent(true, keysPool);
        compositeMeta.addComponent(true, valuesPool);
        compositeMeta.addComponent(true, meta);
        build = true;
        return compositeMeta;
    }

    @Override
    public void close() {
        meta.close();
        valuesPool.close();
        keysPool.close();
    }
}
