package ru.yandex.solomon.slog;

import io.netty.buffer.ByteBuf;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;

/**
 * @author Vladimir Gordiychuk
 */
public class ResolvedLogMetricsHeader {
    /**
     * defines how many bytes used for header by current implementation
     */
    private static final int HEADER_SIZE = 24;
    private static final short SKIP_BYTES = 3;

    /**
     * expected format magic number
     */
    private static final short VALID_MAGIC = 0x5352; // "RS" in LE-order

    /**
     * versions
     */
    public static final short VERSION_1_0 = 0x0100;
    private static final short CURRENT_VERSION = VERSION_1_0; // msb - major, lsb - minor

    public final short version;
    public final int numId;
    public final CompressionAlg compressionAlg;
    public final int labelNamesSize;
    public final int labelValuesSize;
    public final int metricsCount;

    public ResolvedLogMetricsHeader(
        int numId,
        CompressionAlg compressionAlg,
        int labelNamesSize,
        int labelValuesSize,
        int metricsCount)
    {
        this.version = CURRENT_VERSION;
        this.numId = numId;
        this.compressionAlg = compressionAlg;
        this.labelNamesSize = labelNamesSize;
        this.labelValuesSize = labelValuesSize;
        this.metricsCount = metricsCount;
    }

    public int size() {
        return HEADER_SIZE;
    }

    public ResolvedLogMetricsHeader(ByteBuf in) {
        if (in.readableBytes() < HEADER_SIZE) {
            throw new CorruptedBinaryDataRuntimeException(
                "not enough bytes in buffer to read header, " +
                    "need at least: " + HEADER_SIZE + ", but got: " + in.readableBytes());
        }

        final int magic = in.readShortLE();
        if (magic != VALID_MAGIC) {
            throw new CorruptedBinaryDataRuntimeException("invalid magic, expected " +
                Integer.toString(VALID_MAGIC, 16) + ", got " + Integer.toString(magic, 16));
        }

        this.version = in.readShortLE();
        this.numId = in.readIntLE();
        this.labelNamesSize = in.readIntLE();
        this.labelValuesSize = in.readIntLE();
        this.metricsCount = in.readIntLE();
        this.compressionAlg = CompressionAlg.valueOf(in.readByte());
        in.skipBytes(SKIP_BYTES);
    }

    public void writeTo(ByteBuf buffer) {
        buffer
            .writeShortLE(VALID_MAGIC) // 2
            .writeShortLE(version) // 4
            .writeIntLE(numId) // 8
            .writeIntLE(labelNamesSize) // 12
            .writeIntLE(labelValuesSize) // 16
            .writeIntLE(metricsCount) // 20
            .writeByte(compressionAlg.value()) // 21
            .writeZero(SKIP_BYTES); // 24
    }

    @Override
    public String toString() {
        return "ResolvedLogMetricsHeader{" +
            "version=" + version +
            ", numId=" + Integer.toUnsignedLong(numId) +
            ", compressionAlg=" + compressionAlg +
            ", metricsCount=" + metricsCount +
            '}';
    }
}
